#include "chess_comm.h"

#include <cstring>
#include <mutex>
// #include <glog/logging.h>

#define x first
#define y second

using namespace std;
using namespace ChessComm;

ChessHashValuePair g_hash_weight[BORDER_SIZE][BORDER_SIZE];

ChessCoordId g_log2_table[67];
uint64_t g_zobrist_board_hash_weight[4][CHESSBOARD_SIZE];
uint64_t g_zobrist_player_hash_weight[4];

namespace ChessFunction {

bool InBoard(const ChessCoordId id) {
    return 0 <= id && id < CHESSBOARD_SIZE;
}

bool InBoard(const ChessCoordId x, const ChessCoordId y) {
    return 0 <= x && x < BORDER_SIZE
        && 0 <= y && y < BORDER_SIZE;
}

bool IsUnset(const ChessCoordId id) {
    return COORD_UNSET == id;
}

bool IsUnset(const ChessCoordId x, const ChessCoordId y) {
    return COORD_UNSET == CoordToId(x, y);
}

bool IsResign(const ChessCoordId id) {
    return COORD_RESIGN == id;
}

bool IsResign(const ChessCoordId x, const ChessCoordId y) {
    return COORD_RESIGN == CoordToId(x, y);
}


void IdToCoord(const ChessCoordId id, ChessCoordId &x, ChessCoordId &y) {
    if (COORD_RESIGN == id) {
        x = y = COORD_RESIGN;
    } else if (!InBoard(id)) {
        x = y = COORD_UNSET;
    } else {
        x = id / BORDER_SIZE;
        y = id % BORDER_SIZE;
    }
}

ChessCoordId CoordToId(const ChessCoordId x, const ChessCoordId y) {
    if (COORD_RESIGN == x && COORD_RESIGN == y) {
        return COORD_RESIGN;
    }
    if (!InBoard(x, y)) {
        return COORD_UNSET;
    }
    return x * BORDER_SIZE + y;
}

void StrToCoord(const string &str, ChessCoordId &x, ChessCoordId &y) {
    // CHECK_EQ(str.length(), 2) << "string[" << str << "] length not equal to 2";
    x = str[0] - 'a';
    y = str[1] - 'a';
    if (!InBoard(x, y)) {
        x = y = COORD_UNSET;
    }
}

string CoordToStr(const ChessCoordId x, const ChessCoordId y) {
    char buffer[3];
    if (!InBoard(x, y)) {
        buffer[0] = buffer[1] = 'z';
    } else {
        buffer[0] = x + 'a';
        buffer[1] = y + 'a';
    }
    return string(buffer, 2);
}

std::string IdToStr(const ChessCoordId id) {
    ChessCoordId x, y;
    IdToCoord(id, x, y);
    return CoordToStr(x, y);
}

ChessCoordId StrToId(const std::string &str) {
    ChessCoordId x, y;
    StrToCoord(str, x, y);
    return CoordToId(x, y);
}

void ActionToId(const int &action, const ChessCoordId &from_id, const ChessCoordId &to_id, ChessStoneColor &captured_stone, ChessStoneColor &promoted_stone, ChessStoneColor &stone){
    int code[5];
    int i;
    if(action == COORD_RESIGN){
        from_id = to_id = COORD_RESIGN;
        captured_stone = promoted_stone = stone = EMPTY;
    }else if(action == COORD_UNSET){
        from_id = to_id = COORD_UNSET;
        captured_stone = promoted_stone = stone = EMPTY;
    }else{
        for(i = 0; i < 5; i++){
            code[i] = 0;
        }
        for(i = 0; i < 5; i++){
            code[i] = action % 64;
            action = action / 64;
        }
        from_id = code[4];
        to_id = code[3];
        captured_stone = code[2];
        promoted_stone = code[1];
        stone = code[0];
    }
}

void IdToAction(const ChessCoordId &from_id, const ChessCoordId &to_id, ChessStoneColor &captured_stone, ChessStoneColor &promoted_stone, ChessStoneColor &stone, const int &action){
    int code[5];
    int i, j;
    for(i = 0; i < 5; i++){
        code[i] = 1;
    }
    if(from_id == COORD_RESIGN || to_id == COORD_RESIGN){
        action = COORD_RESIGN;
    }else if(!InBoard(from_id) || !InBoard(to_id)){
        action = COORD_UNSET;
    }else if(from_id >= 0 && to_id >= 0 && captured_stone >= 0 && promoted_stone >= 0 && stone >= 0){
        for(i = 0; i < 5; i++){
            for(j = i; j > 0; j--){
                code[i] *= 64;
            }
        }
        action = from_id * code[4] + to_id * code[3] + captured_stone * code[2] + promoted_stone * code[1] + stone * code[0];
    }
}

once_flag CreateGlobalVariables_once;
void CreateGlobalVariables() {
    call_once(
        CreateGlobalVariables_once,
        []() {
            CreateHashWeights();
            CreateQuickLog2Table();
            CreateZobristHash();
        }
    );
}


void CreateHashWeights() {
    g_hash_weight[0][0] = ChessHashValuePair(1, 1);
    for (ChessCoordId i = 1; i < CHESSBOARD_SIZE; ++i) {
        g_hash_weight[i / BORDER_SIZE][i % BORDER_SIZE] =
            ChessHashValuePair(g_hash_weight[(i - 1) / BORDER_SIZE][(i - 1) % BORDER_SIZE].x * g_hash_unit.x,
                            g_hash_weight[(i - 1) / BORDER_SIZE][(i - 1) % BORDER_SIZE].y * g_hash_unit.y);
    }
}

void CreateQuickLog2Table() {
    memset(g_log2_table, -1, sizeof(g_log2_table));
    int tmp = 1;

    for (ChessCoordId i = 0; i < 64; ++i) {
        g_log2_table[tmp] = i;
        tmp *= 2;
        tmp %= 67;
    }
}

#if defined(_WIN32) || defined(_WIN64)
static int rand_r(unsigned int *seed)
{
    unsigned int next = *seed;
    int result;

    next *= 1103515245;
    next += 12345;
    result = (unsigned int)(next / 65536) % 2048;

    next *= 1103515245;
    next += 12345;
    result <<= 10;
    result ^= (unsigned int)(next / 65536) % 1024;

    next *= 1103515245;
    next += 12345;
    result <<= 10;
    result ^= (unsigned int)(next / 65536) % 1024;

    *seed = next;

    return result;
}
#endif

void CreateZobristHash() {
    uint32_t seed = 0xdeadbeaf;

    for (int i = 0; i < 4; ++i) {
        g_zobrist_player_hash_weight[i] = (uint64_t) rand_r(&seed) << 32 | rand_r(&seed);
        for (int j = 0; j < CHESSBOARD_SIZE; ++j) {
            g_zobrist_board_hash_weight[i][j] = (uint64_t) rand_r(&seed) << 32 | rand_r(&seed);
        }
    }

}

} // namespace ChessFunction

#undef y
#undef x
