import re
import codecritic.dataset.code as codelib


def get_hint_prompt(question, incorrect_code, diff_edit, level):
    prompt = f"""
You are a helpful programming assistant. Your task is to analyze the following **question, incorrect code, and diff edit**. Think step by step and generate a hint that:

1. Explains **why the algorithm is incorrect** (algorithmic reasoning).
2. Provides **actionable guidance** on how to fix the code (code action).

Ensure the hint is clear, actionable, and appropriate for a **{level}-level** learner.

---

## Input

### Question
{question}

### Incorrect Code
```python
{incorrect_code}
```

### Diff Edit
```diff
{diff_edit}
```

---

## Output Format
Return your response in the following format:

### Hint
[Your hint here. Include both algorithmic reasoning and actionable guidance. Natural Language Only.]
"""
    return prompt.strip()


def process_to_hint_prompt(chosen, rejected, level):
    question = chosen["messages"][0]["content"]
    # question = "\n".join(question.strip().splitlines()[1:-1])

    chosen_code = codelib.preprocess_code(chosen["code"])
    rejected_code = codelib.preprocess_code(rejected["code"])

    diff_str = codelib.diff_code(rejected_code, chosen_code)
    prompt = get_hint_prompt(question, rejected_code, diff_str, level)
    messages = [{"role": "user", "content": prompt}]
    return {
        "dataset": rejected["dataset"],
        "task_id": rejected["task_id"],
        "solution_id": rejected["solution_id"],
        "messages": messages
    }


hint_pattern = re.compile(r"### Hint\n(.*?)(?=\n###|$)", re.DOTALL)
def postprocess_to_hint(llm_response):
    messages = llm_response.pop("messages")
    response = messages[-1]["content"]
    hint_match = hint_pattern.search(response)
    if hint_match:
        hint = hint_match.group(1).strip()
    else:
        hint = "No hints."
        print("No hints in response: {}".format(response))

    llm_response["hint"] = hint
    return llm_response


def get_reason_prompt(question, code, hint):
    prompt = f"""
You are a code analysis assistant specialized in finding bugs.
When reviewing code, reason through its logic step by step:

- Analyze each part's purpose and implementation
- If you spot potential issues, explain:
  * Why it's a problem (based on the logic)
  * How to fix it

---

## Input

### Question
{question}

### Code
```python
{code}
```
"""
    hint_prompt = f"""

### Hint
{hint}
"""
    if hint is None:
        return prompt.strip()
    else:
        return (prompt + hint_prompt).strip()


def process_to_reason_prompt(item, hint):
    question = item["messages"][0]["content"]
    code = item["code"]

    prompt = get_reason_prompt(question, code, hint)
    messages = [{"role": "user", "content": prompt}]
    return {
        "dataset": item["dataset"],
        "task_id": item["task_id"],
        "solution_id": item["solution_id"],
        "messages": messages,
        "pass": item["pass"]
    }


def get_debug_prompt():
    return """
Based on the analysis provided, please:

1. **Draw a conclusion**: State whether the original code is correct or not by answering "Yes" or "No".
   - Format: `Conclusion: <Yes/No>`

2. **If the code is not correct**, provide the corrected code.

---

## Output format
Conclusion: (Yes/No)
Corrected Code:
```python
[corrected code here]
```
""".strip()


def remove_hint(item):
    raw_prompt = item["messages"][0]["content"]
    hint_start = raw_prompt.find("### Hint")
    if hint_start != -1:
        truncated_string = raw_prompt[:hint_start].strip()
    else:
        truncated_string = raw_prompt
    item["messages"][0]["content"] = truncated_string


def extract_conclusion_and_code(response):
    # Extract conclusion
    try:
        conclusion_line = [line for line in response.split('\n') if line.startswith('Conclusion:')][0]
        conclusion_str = conclusion_line.split(': ')[1].strip().lower()

        if "yes" in conclusion_str:
            conclusion = True
        elif "no" in conclusion_str:
            conclusion = False
        else:
            print("llm doesn't draw to a conclusion\n{}".format(response))
            conclusion = None
    except Exception as e:
        print("not found conclusion\n{}\n{}".format(response, e))
        conclusion = None

    # Extract corrected code if conclusion is 'No'
    corrected_code = ""
    if not conclusion:
        corrected_code = codelib.extract_code(response)

    return conclusion, corrected_code
