import argparse
from functools import partial
from pathlib import Path

import codecritic.evaluation.metric as metric
from codecritic.utils.json import load_jsonl, save_jsonl

def eval(samples_path):
    model, testset = samples_path.stem.split('-')[:2]
    def f(item):
        item["model"] = model
        item["testset"] = testset
    
    samples = load_jsonl(samples_path)

    ks = list(range(1, 17))

    results = []
    results.append(metric.pass_at_k(samples, ks))
    results.append(metric.top_at_k(samples, ks, metric.postive_and_negative))
    results.append(metric.top_at_k(samples, ks, metric.positive_only))

    for i in range(4):
        threshold = 0.5 + i * 0.1
        score_func = partial(metric.pos_neg_filter_uncertain, threshold=threshold)
        results.append(metric.top_at_k(samples, ks, score_func))

    return list(map(f, results))


if __name__ == "__main__":
    parser = argparse.ArgumentParser()
    parser.add_argument(
        "--sample_dir",
        type=str,
        default=None,
        help="Path to the directory containing samples. If not provided, cached results will be used."
    )
    parser.add_argument("--out_dir", type=str, help="path/to/output_dir")
    parser.add_argument(
        "--score_func",
        type=str,
        default="all",
        choices=["all", "posonly", "posneg", "posneg_filter"],  # Add valid options
        help="Select the scoring function to use. Default: 'all'."
    )
    parser.add_argument("--plot", type=str, help="path/to/plot")
    args = parser.parse_args()

    outdir = Path(args.out_dir)
    if args.sample_dir:
        for samples_path in Path(args.sample_dir).glob("*.jsonl"):
            out_path = outdir / (samples_path.stem + "-eval.jsonl")
            if not out_path.exists():
                eval_results = eval(samples_path)
                save_jsonl(eval_results, out_path)
    
    for out_path in outdir.glob("*.jsonl"):
        pass