#include "clar_libgit2.h"
#include "checkout_helpers.h"

#include "git2/checkout.h"
#include "repository.h"

#define UTF8_BOM "\xEF\xBB\xBF"
#define ALL_CRLF_TEXT_RAW	"crlf\r\ncrlf\r\ncrlf\r\ncrlf\r\n"
#define ALL_LF_TEXT_RAW		"lf\nlf\nlf\nlf\nlf\n"
#define MORE_CRLF_TEXT_RAW	"crlf\r\ncrlf\r\nlf\ncrlf\r\ncrlf\r\n"
#define MORE_LF_TEXT_RAW	"lf\nlf\ncrlf\r\nlf\nlf\n"

#define ALL_LF_TEXT_AS_CRLF	"lf\r\nlf\r\nlf\r\nlf\r\nlf\r\n"

static git_repository *g_repo;

void test_checkout_crlf__initialize(void)
{
	git_tree *tree;

	g_repo = cl_git_sandbox_init("crlf");

	cl_git_pass(git_repository_head_tree(&tree, g_repo));
}

void test_checkout_crlf__cleanup(void)
{
	cl_git_sandbox_cleanup();
}

static void set_config_entry_to(const char *entry_name, bool value)
{
	git_config *cfg;

	cl_git_pass(git_repository_config(&cfg, g_repo));
	cl_git_pass(git_config_set_bool(cfg, entry_name, value));

	git_config_free(cfg);
}

static void set_core_autocrlf_to(bool value)
{
	set_config_entry_to("core.autocrlf", value);
}

void test_checkout_crlf__detect_crlf_autocrlf_false(void)
{
	git_checkout_opts opts = GIT_CHECKOUT_OPTS_INIT;
	opts.checkout_strategy = GIT_CHECKOUT_SAFE_CREATE;

	set_core_autocrlf_to(false);

	git_checkout_head(g_repo, &opts);

	test_file_contents("./crlf/all-lf", ALL_LF_TEXT_RAW);
}

void test_checkout_crlf__autocrlf_false_index_size_is_unfiltered_size(void)
{
	git_index *index;
	const git_index_entry *entry;
	git_checkout_opts opts = GIT_CHECKOUT_OPTS_INIT;
	opts.checkout_strategy = GIT_CHECKOUT_SAFE_CREATE;

	set_core_autocrlf_to(false);

	git_checkout_head(g_repo, &opts);

	git_repository_index(&index, g_repo);

	cl_assert((entry = git_index_get_bypath(index, "all-lf", 0)) != NULL);
	cl_assert(entry->file_size == strlen(ALL_LF_TEXT_RAW));

	git_index_free(index);
}

void test_checkout_crlf__detect_crlf_autocrlf_true(void)
{
	git_checkout_opts opts = GIT_CHECKOUT_OPTS_INIT;
	opts.checkout_strategy = GIT_CHECKOUT_SAFE_CREATE;

	set_core_autocrlf_to(true);

	git_checkout_head(g_repo, &opts);

	test_file_contents("./crlf/all-lf", ALL_LF_TEXT_AS_CRLF);
}

void test_checkout_crlf__autocrlf_true_index_size_is_filtered_size(void)
{
	git_index *index;
	const git_index_entry *entry;
	git_checkout_opts opts = GIT_CHECKOUT_OPTS_INIT;
	opts.checkout_strategy = GIT_CHECKOUT_SAFE_CREATE;

	set_core_autocrlf_to(true);

	git_checkout_head(g_repo, &opts);

	git_repository_index(&index, g_repo);

	cl_assert((entry = git_index_get_bypath(index, "all-lf", 0)) != NULL);
	cl_assert(entry->file_size == strlen(ALL_LF_TEXT_AS_CRLF));

	git_index_free(index);
}
