#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>

#include "test_lib.h"
#include <git/odb.h>
#include "fileops.h"

static char *odb_dir = "test-objects";

typedef struct object_data {
    unsigned char *bytes;  /* (compressed) bytes stored in object store */
    size_t        blen;    /* length of data in object store            */
    char          *id;     /* object id (sha1)                          */
    char          *type;   /* object type                               */
    char          *dir;    /* object store (fan-out) directory name     */
    char          *file;   /* object store filename                     */
    unsigned char *data;   /* (uncompressed) object data                */
    size_t        dlen;    /* length of (uncompressed) object data      */
} object_data;

/* commit == 3d7f8a6af076c8c3f20071a8935cdbe8228594d1 */
static unsigned char commit_bytes[] = {
    0x78, 0x01, 0x85, 0x50, 0xc1, 0x6a, 0xc3, 0x30,
    0x0c, 0xdd, 0xd9, 0x5f, 0xa1, 0xfb, 0x96, 0x12,
    0xbb, 0x29, 0x71, 0x46, 0x19, 0x2b, 0x3d, 0x97,
    0x1d, 0xd6, 0x7d, 0x80, 0x1d, 0xcb, 0x89, 0x21,
    0xb6, 0x82, 0xed, 0x40, 0xf3, 0xf7, 0xf3, 0x48,
    0x29, 0x3b, 0x6d, 0xd2, 0xe5, 0xbd, 0x27, 0xbd,
    0x27, 0x50, 0x4f, 0xde, 0xbb, 0x0c, 0xfb, 0x43,
    0xf3, 0x94, 0x23, 0x22, 0x18, 0x6b, 0x85, 0x51,
    0x5d, 0xad, 0xc5, 0xa1, 0x41, 0xae, 0x51, 0x4b,
    0xd9, 0x19, 0x6e, 0x4b, 0x0b, 0x29, 0x35, 0x72,
    0x59, 0xef, 0x5b, 0x29, 0x8c, 0x65, 0x6a, 0xc9,
    0x23, 0x45, 0x38, 0xc1, 0x17, 0x5c, 0x7f, 0xc0,
    0x71, 0x13, 0xde, 0xf1, 0xa6, 0xfc, 0x3c, 0xe1,
    0xae, 0x27, 0xff, 0x06, 0x5c, 0x88, 0x56, 0xf2,
    0x46, 0x74, 0x2d, 0x3c, 0xd7, 0xa5, 0x58, 0x51,
    0xcb, 0xb9, 0x8c, 0x11, 0xce, 0xf0, 0x01, 0x97,
    0x0d, 0x1e, 0x1f, 0xea, 0x3f, 0x6e, 0x76, 0x02,
    0x0a, 0x58, 0x4d, 0x2e, 0x20, 0x6c, 0x1e, 0x48,
    0x8b, 0xf7, 0x2a, 0xae, 0x8c, 0x5d, 0x47, 0x04,
    0x4d, 0x66, 0x05, 0xb2, 0x90, 0x0b, 0xbe, 0xcf,
    0x3d, 0xa6, 0xa4, 0x06, 0x7c, 0x29, 0x3c, 0x64,
    0xe5, 0x82, 0x0b, 0x03, 0xd8, 0x25, 0x96, 0x8d,
    0x08, 0x78, 0x9b, 0x27, 0x15, 0x54, 0x76, 0x14,
    0xd8, 0xdd, 0x35, 0x2f, 0x71, 0xa6, 0x84, 0x8f,
    0x90, 0x51, 0x85, 0x01, 0x13, 0xb8, 0x90, 0x23,
    0x99, 0xa5, 0x47, 0x03, 0x7a, 0xfd, 0x15, 0xbf,
    0x63, 0xec, 0xd3, 0x0d, 0x01, 0x4d, 0x45, 0xb6,
    0xd2, 0xeb, 0xeb, 0xdf, 0xef, 0x60, 0xdf, 0xef,
    0x1f, 0x78, 0x35,
};

static unsigned char commit_data[] = {
    0x74, 0x72, 0x65, 0x65, 0x20, 0x64, 0x66, 0x66,
    0x32, 0x64, 0x61, 0x39, 0x30, 0x62, 0x32, 0x35,
    0x34, 0x65, 0x31, 0x62, 0x65, 0x62, 0x38, 0x38,
    0x39, 0x64, 0x31, 0x66, 0x31, 0x66, 0x31, 0x32,
    0x38, 0x38, 0x62, 0x65, 0x31, 0x38, 0x30, 0x33,
    0x37, 0x38, 0x32, 0x64, 0x66, 0x0a, 0x61, 0x75,
    0x74, 0x68, 0x6f, 0x72, 0x20, 0x41, 0x20, 0x55,
    0x20, 0x54, 0x68, 0x6f, 0x72, 0x20, 0x3c, 0x61,
    0x75, 0x74, 0x68, 0x6f, 0x72, 0x40, 0x65, 0x78,
    0x61, 0x6d, 0x70, 0x6c, 0x65, 0x2e, 0x63, 0x6f,
    0x6d, 0x3e, 0x20, 0x31, 0x32, 0x32, 0x37, 0x38,
    0x31, 0x34, 0x32, 0x39, 0x37, 0x20, 0x2b, 0x30,
    0x30, 0x30, 0x30, 0x0a, 0x63, 0x6f, 0x6d, 0x6d,
    0x69, 0x74, 0x74, 0x65, 0x72, 0x20, 0x43, 0x20,
    0x4f, 0x20, 0x4d, 0x69, 0x74, 0x74, 0x65, 0x72,
    0x20, 0x3c, 0x63, 0x6f, 0x6d, 0x6d, 0x69, 0x74,
    0x74, 0x65, 0x72, 0x40, 0x65, 0x78, 0x61, 0x6d,
    0x70, 0x6c, 0x65, 0x2e, 0x63, 0x6f, 0x6d, 0x3e,
    0x20, 0x31, 0x32, 0x32, 0x37, 0x38, 0x31, 0x34,
    0x32, 0x39, 0x37, 0x20, 0x2b, 0x30, 0x30, 0x30,
    0x30, 0x0a, 0x0a, 0x41, 0x20, 0x6f, 0x6e, 0x65,
    0x2d, 0x6c, 0x69, 0x6e, 0x65, 0x20, 0x63, 0x6f,
    0x6d, 0x6d, 0x69, 0x74, 0x20, 0x73, 0x75, 0x6d,
    0x6d, 0x61, 0x72, 0x79, 0x0a, 0x0a, 0x54, 0x68,
    0x65, 0x20, 0x62, 0x6f, 0x64, 0x79, 0x20, 0x6f,
    0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x63, 0x6f,
    0x6d, 0x6d, 0x69, 0x74, 0x20, 0x6d, 0x65, 0x73,
    0x73, 0x61, 0x67, 0x65, 0x2c, 0x20, 0x63, 0x6f,
    0x6e, 0x74, 0x61, 0x69, 0x6e, 0x69, 0x6e, 0x67,
    0x20, 0x66, 0x75, 0x72, 0x74, 0x68, 0x65, 0x72,
    0x20, 0x65, 0x78, 0x70, 0x6c, 0x61, 0x6e, 0x61,
    0x74, 0x69, 0x6f, 0x6e, 0x0a, 0x6f, 0x66, 0x20,
    0x74, 0x68, 0x65, 0x20, 0x70, 0x75, 0x72, 0x70,
    0x6f, 0x73, 0x65, 0x20, 0x6f, 0x66, 0x20, 0x74,
    0x68, 0x65, 0x20, 0x63, 0x68, 0x61, 0x6e, 0x67,
    0x65, 0x73, 0x20, 0x69, 0x6e, 0x74, 0x72, 0x6f,
    0x64, 0x75, 0x63, 0x65, 0x64, 0x20, 0x62, 0x79,
    0x20, 0x74, 0x68, 0x65, 0x20, 0x63, 0x6f, 0x6d,
    0x6d, 0x69, 0x74, 0x2e, 0x0a, 0x0a, 0x53, 0x69,
    0x67, 0x6e, 0x65, 0x64, 0x2d, 0x6f, 0x66, 0x2d,
    0x62, 0x79, 0x3a, 0x20, 0x41, 0x20, 0x55, 0x20,
    0x54, 0x68, 0x6f, 0x72, 0x20, 0x3c, 0x61, 0x75,
    0x74, 0x68, 0x6f, 0x72, 0x40, 0x65, 0x78, 0x61,
    0x6d, 0x70, 0x6c, 0x65, 0x2e, 0x63, 0x6f, 0x6d,
    0x3e, 0x0a,
};

static object_data commit = {
    commit_bytes,
    sizeof(commit_bytes),
    "3d7f8a6af076c8c3f20071a8935cdbe8228594d1",
    "commit",
    "test-objects/3d",
    "test-objects/3d/7f8a6af076c8c3f20071a8935cdbe8228594d1",
    commit_data,
    sizeof(commit_data),
};

/* tree == dff2da90b254e1beb889d1f1f1288be1803782df */
static unsigned char tree_bytes[] = {
    0x78, 0x01, 0x2b, 0x29, 0x4a, 0x4d, 0x55, 0x30,
    0x34, 0x32, 0x63, 0x30, 0x34, 0x30, 0x30, 0x33,
    0x31, 0x51, 0xc8, 0xcf, 0x4b, 0x65, 0xe8, 0x16,
    0xae, 0x98, 0x58, 0x29, 0xff, 0x32, 0x53, 0x7d,
    0x6d, 0xc5, 0x33, 0x6f, 0xae, 0xb5, 0xd5, 0xf7,
    0x2e, 0x74, 0xdf, 0x81, 0x4a, 0x17, 0xe7, 0xe7,
    0xa6, 0x32, 0xfc, 0x6d, 0x31, 0xd8, 0xd3, 0xe6,
    0xf3, 0xe7, 0xea, 0x47, 0xbe, 0xd0, 0x09, 0x3f,
    0x96, 0xb8, 0x3f, 0x90, 0x9e, 0xa2, 0xfd, 0x0f,
    0x2a, 0x5f, 0x52, 0x9e, 0xcf, 0x50, 0x31, 0x43,
    0x52, 0x29, 0xd1, 0x5a, 0xeb, 0x77, 0x82, 0x2a,
    0x8b, 0xfe, 0xb7, 0xbd, 0xed, 0x5d, 0x07, 0x67,
    0xfa, 0xb5, 0x42, 0xa5, 0xab, 0x52, 0x8b, 0xf2,
    0x19, 0x9e, 0xcd, 0x7d, 0x34, 0x7b, 0xd3, 0xc5,
    0x6b, 0xce, 0xde, 0xdd, 0x9a, 0xeb, 0xca, 0xa3,
    0x6e, 0x1c, 0x7a, 0xd2, 0x13, 0x3c, 0x11, 0x00,
    0xe2, 0xaa, 0x38, 0x57,
};

static unsigned char tree_data[] = {
    0x31, 0x30, 0x30, 0x36, 0x34, 0x34, 0x20, 0x6f,
    0x6e, 0x65, 0x00, 0x8b, 0x13, 0x78, 0x91, 0x79,
    0x1f, 0xe9, 0x69, 0x27, 0xad, 0x78, 0xe6, 0x4b,
    0x0a, 0xad, 0x7b, 0xde, 0xd0, 0x8b, 0xdc, 0x31,
    0x30, 0x30, 0x36, 0x34, 0x34, 0x20, 0x73, 0x6f,
    0x6d, 0x65, 0x00, 0xfd, 0x84, 0x30, 0xbc, 0x86,
    0x4c, 0xfc, 0xd5, 0xf1, 0x0e, 0x55, 0x90, 0xf8,
    0xa4, 0x47, 0xe0, 0x1b, 0x94, 0x2b, 0xfe, 0x31,
    0x30, 0x30, 0x36, 0x34, 0x34, 0x20, 0x74, 0x77,
    0x6f, 0x00, 0x78, 0x98, 0x19, 0x22, 0x61, 0x3b,
    0x2a, 0xfb, 0x60, 0x25, 0x04, 0x2f, 0xf6, 0xbd,
    0x87, 0x8a, 0xc1, 0x99, 0x4e, 0x85, 0x31, 0x30,
    0x30, 0x36, 0x34, 0x34, 0x20, 0x7a, 0x65, 0x72,
    0x6f, 0x00, 0xe6, 0x9d, 0xe2, 0x9b, 0xb2, 0xd1,
    0xd6, 0x43, 0x4b, 0x8b, 0x29, 0xae, 0x77, 0x5a,
    0xd8, 0xc2, 0xe4, 0x8c, 0x53, 0x91,
};

static object_data tree = {
    tree_bytes,
    sizeof(tree_bytes),
    "dff2da90b254e1beb889d1f1f1288be1803782df",
    "tree",
    "test-objects/df",
    "test-objects/df/f2da90b254e1beb889d1f1f1288be1803782df",
    tree_data,
    sizeof(tree_data),
};

/* tag == 09d373e1dfdc16b129ceec6dd649739911541e05 */
static unsigned char tag_bytes[] = {
    0x78, 0x01, 0x35, 0x4e, 0xcb, 0x0a, 0xc2, 0x40,
    0x10, 0xf3, 0xbc, 0x5f, 0x31, 0x77, 0xa1, 0xec,
    0xa3, 0xed, 0x6e, 0x41, 0x44, 0xf0, 0x2c, 0x5e,
    0xfc, 0x81, 0xe9, 0x76, 0xb6, 0xad, 0xb4, 0xb4,
    0x6c, 0x07, 0xd1, 0xbf, 0x77, 0x44, 0x0d, 0x39,
    0x84, 0x10, 0x92, 0x30, 0xf6, 0x60, 0xbc, 0xdb,
    0x2d, 0xed, 0x9d, 0x22, 0x83, 0xeb, 0x7c, 0x0a,
    0x58, 0x63, 0xd2, 0xbe, 0x8e, 0x21, 0xba, 0x64,
    0xb5, 0xf6, 0x06, 0x43, 0xe3, 0xaa, 0xd8, 0xb5,
    0x14, 0xac, 0x0d, 0x55, 0x53, 0x76, 0x46, 0xf1,
    0x6b, 0x25, 0x88, 0xcb, 0x3c, 0x8f, 0xac, 0x58,
    0x3a, 0x1e, 0xba, 0xd0, 0x85, 0xd8, 0xd8, 0xf7,
    0x94, 0xe1, 0x0c, 0x57, 0xb8, 0x8c, 0xcc, 0x22,
    0x0f, 0xdf, 0x90, 0xc8, 0x13, 0x3d, 0x71, 0x5e,
    0x27, 0x2a, 0xc4, 0x39, 0x82, 0xb1, 0xd6, 0x07,
    0x53, 0xda, 0xc6, 0xc3, 0x5e, 0x0b, 0x94, 0xba,
    0x0d, 0xe3, 0x06, 0x42, 0x1e, 0x08, 0x3e, 0x95,
    0xbf, 0x4b, 0x69, 0xc9, 0x90, 0x69, 0x22, 0xdc,
    0xe8, 0xbf, 0xf2, 0x06, 0x42, 0x9a, 0x36, 0xb1,
};

static unsigned char tag_data[] = {
    0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x20, 0x33,
    0x64, 0x37, 0x66, 0x38, 0x61, 0x36, 0x61, 0x66,
    0x30, 0x37, 0x36, 0x63, 0x38, 0x63, 0x33, 0x66,
    0x32, 0x30, 0x30, 0x37, 0x31, 0x61, 0x38, 0x39,
    0x33, 0x35, 0x63, 0x64, 0x62, 0x65, 0x38, 0x32,
    0x32, 0x38, 0x35, 0x39, 0x34, 0x64, 0x31, 0x0a,
    0x74, 0x79, 0x70, 0x65, 0x20, 0x63, 0x6f, 0x6d,
    0x6d, 0x69, 0x74, 0x0a, 0x74, 0x61, 0x67, 0x20,
    0x76, 0x30, 0x2e, 0x30, 0x2e, 0x31, 0x0a, 0x74,
    0x61, 0x67, 0x67, 0x65, 0x72, 0x20, 0x43, 0x20,
    0x4f, 0x20, 0x4d, 0x69, 0x74, 0x74, 0x65, 0x72,
    0x20, 0x3c, 0x63, 0x6f, 0x6d, 0x6d, 0x69, 0x74,
    0x74, 0x65, 0x72, 0x40, 0x65, 0x78, 0x61, 0x6d,
    0x70, 0x6c, 0x65, 0x2e, 0x63, 0x6f, 0x6d, 0x3e,
    0x20, 0x31, 0x32, 0x32, 0x37, 0x38, 0x31, 0x34,
    0x32, 0x39, 0x37, 0x20, 0x2b, 0x30, 0x30, 0x30,
    0x30, 0x0a, 0x0a, 0x54, 0x68, 0x69, 0x73, 0x20,
    0x69, 0x73, 0x20, 0x74, 0x68, 0x65, 0x20, 0x74,
    0x61, 0x67, 0x20, 0x6f, 0x62, 0x6a, 0x65, 0x63,
    0x74, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x72, 0x65,
    0x6c, 0x65, 0x61, 0x73, 0x65, 0x20, 0x76, 0x30,
    0x2e, 0x30, 0x2e, 0x31, 0x0a,
};

static object_data tag = {
    tag_bytes,
    sizeof(tag_bytes),
    "09d373e1dfdc16b129ceec6dd649739911541e05",
    "tag",
    "test-objects/09",
    "test-objects/09/d373e1dfdc16b129ceec6dd649739911541e05",
    tag_data,
    sizeof(tag_data),
};

/* zero == e69de29bb2d1d6434b8b29ae775ad8c2e48c5391 */
static unsigned char zero_bytes[] = {
    0x78, 0x01, 0x4b, 0xca, 0xc9, 0x4f, 0x52, 0x30,
    0x60, 0x00, 0x00, 0x09, 0xb0, 0x01, 0xf0,
};

static unsigned char zero_data[] = {
    0x00  /* dummy data */
};

static object_data zero = {
    zero_bytes,
    sizeof(zero_bytes),
    "e69de29bb2d1d6434b8b29ae775ad8c2e48c5391",
    "blob",
    "test-objects/e6",
    "test-objects/e6/9de29bb2d1d6434b8b29ae775ad8c2e48c5391",
    zero_data,
    0,
};

/* one == 8b137891791fe96927ad78e64b0aad7bded08bdc */
static unsigned char one_bytes[] = {
    0x78, 0x01, 0x4b, 0xca, 0xc9, 0x4f, 0x52, 0x30,
    0x64, 0xe0, 0x02, 0x00, 0x0b, 0xad, 0x01, 0xfb,
};

static unsigned char one_data[] = {
    0x0a,
};

static object_data one = {
    one_bytes,
    sizeof(one_bytes),
    "8b137891791fe96927ad78e64b0aad7bded08bdc",
    "blob",
    "test-objects/8b",
    "test-objects/8b/137891791fe96927ad78e64b0aad7bded08bdc",
    one_data,
    sizeof(one_data),
};

/* two == 78981922613b2afb6025042ff6bd878ac1994e85 */
static unsigned char two_bytes[] = {
    0x78, 0x01, 0x4b, 0xca, 0xc9, 0x4f, 0x52, 0x30,
    0x62, 0x48, 0xe4, 0x02, 0x00, 0x0e, 0x64, 0x02,
    0x5d,
};

static unsigned char two_data[] = {
    0x61, 0x0a,
};

static object_data two = {
    two_bytes,
    sizeof(two_bytes),
    "78981922613b2afb6025042ff6bd878ac1994e85",
    "blob",
    "test-objects/78",
    "test-objects/78/981922613b2afb6025042ff6bd878ac1994e85",
    two_data,
    sizeof(two_data),
};

/* some == fd8430bc864cfcd5f10e5590f8a447e01b942bfe */
static unsigned char some_bytes[] = {
    0x78, 0x01, 0x7d, 0x54, 0xc1, 0x4e, 0xe3, 0x30,
    0x10, 0xdd, 0x33, 0x5f, 0x31, 0xc7, 0x5d, 0x94,
    0xa5, 0x84, 0xd5, 0x22, 0xad, 0x7a, 0x0a, 0x15,
    0x85, 0x48, 0xd0, 0x56, 0x49, 0x2a, 0xd4, 0xa3,
    0x13, 0x4f, 0x88, 0x85, 0x63, 0x47, 0xb6, 0x43,
    0xc9, 0xdf, 0xef, 0x8c, 0x69, 0x17, 0x56, 0x0b,
    0x7b, 0xaa, 0x62, 0x7b, 0xde, 0xbc, 0xf7, 0xe6,
    0x4d, 0x6b, 0x6d, 0x6b, 0x48, 0xd3, 0xcb, 0x5f,
    0x5f, 0x66, 0xa7, 0x27, 0x70, 0x0a, 0x55, 0xa7,
    0x3c, 0xb4, 0x4a, 0x23, 0xf0, 0xaf, 0x43, 0x04,
    0x6f, 0xdb, 0xb0, 0x17, 0x0e, 0xe7, 0x30, 0xd9,
    0x11, 0x1a, 0x61, 0xc0, 0xa1, 0x54, 0x3e, 0x38,
    0x55, 0x8f, 0x81, 0x9e, 0x05, 0x10, 0x46, 0xce,
    0xac, 0x83, 0xde, 0x4a, 0xd5, 0x4e, 0x0c, 0x42,
    0x67, 0xa3, 0x91, 0xe8, 0x20, 0x74, 0x08, 0x01,
    0x5d, 0xef, 0xc1, 0xb6, 0xf1, 0xe3, 0x66, 0xb5,
    0x85, 0x1b, 0x34, 0xe8, 0x84, 0x86, 0xcd, 0x58,
    0x6b, 0xd5, 0xc0, 0x9d, 0x6a, 0xd0, 0x78, 0x4c,
    0xe0, 0x19, 0x9d, 0x57, 0xd6, 0xc0, 0x45, 0xc2,
    0x18, 0xc2, 0xc3, 0xc0, 0x0f, 0x7c, 0x87, 0x12,
    0xea, 0x29, 0x56, 0x2f, 0x99, 0x4f, 0x79, 0xe0,
    0x03, 0x4b, 0x4b, 0x4d, 0x44, 0xa0, 0x92, 0x33,
    0x2a, 0xe0, 0x9a, 0xdc, 0x80, 0x90, 0x52, 0xf1,
    0x11, 0x04, 0x1b, 0x4b, 0x06, 0xea, 0xae, 0x3c,
    0xe3, 0x7a, 0x50, 0x74, 0x4a, 0x84, 0xfe, 0xc3,
    0x81, 0x41, 0xf8, 0x89, 0x18, 0x43, 0x67, 0x9d,
    0x87, 0x47, 0xf5, 0x8c, 0x51, 0xf6, 0x68, 0xb4,
    0xea, 0x55, 0x20, 0x2a, 0x6f, 0x80, 0xdc, 0x42,
    0x2b, 0xf3, 0x14, 0x2b, 0x1a, 0xdb, 0x0f, 0xe4,
    0x9a, 0x64, 0x84, 0xa3, 0x90, 0xa8, 0xf9, 0x8f,
    0x9d, 0x86, 0x9e, 0xd3, 0xab, 0x5a, 0x99, 0xc8,
    0xd9, 0xc3, 0x5e, 0x85, 0x0e, 0x2c, 0xb5, 0x73,
    0x30, 0x38, 0xfb, 0xe8, 0x44, 0xef, 0x5f, 0x95,
    0x1b, 0xc9, 0xd0, 0xef, 0x3c, 0x26, 0x32, 0x1e,
    0xff, 0x2d, 0xb6, 0x23, 0x7b, 0x3f, 0xd1, 0x3c,
    0x78, 0x1a, 0x0d, 0xcb, 0xe6, 0xf6, 0xd4, 0x44,
    0x99, 0x47, 0x1a, 0x9e, 0xed, 0x23, 0xb5, 0x91,
    0x6a, 0xdf, 0x53, 0x39, 0x03, 0xf8, 0x5a, 0xb1,
    0x0f, 0x1f, 0xce, 0x81, 0x11, 0xde, 0x01, 0x7a,
    0x90, 0x16, 0xc4, 0x30, 0xe8, 0x89, 0xed, 0x7b,
    0x65, 0x4b, 0xd7, 0x03, 0x36, 0xc1, 0xcf, 0xa1,
    0xa5, 0xb1, 0xe3, 0x8b, 0xe8, 0x07, 0x4d, 0xf3,
    0x23, 0x25, 0x13, 0x35, 0x27, 0xf5, 0x8c, 0x11,
    0xd3, 0xa0, 0x9a, 0xa8, 0xf5, 0x38, 0x7d, 0xce,
    0x55, 0xc2, 0x71, 0x79, 0x13, 0xc7, 0xa3, 0xda,
    0x77, 0x68, 0xc0, 0xd8, 0x10, 0xdd, 0x24, 0x8b,
    0x15, 0x59, 0xc5, 0x10, 0xe2, 0x20, 0x99, 0x8e,
    0xf0, 0x05, 0x9b, 0x31, 0x88, 0x5a, 0xe3, 0xd9,
    0x37, 0xba, 0xe2, 0xdb, 0xbf, 0x92, 0xfa, 0x66,
    0x16, 0x97, 0x47, 0xd9, 0x9d, 0x1d, 0x28, 0x7c,
    0x9d, 0x08, 0x1c, 0xc7, 0xbd, 0xd2, 0x1a, 0x6a,
    0x04, 0xf2, 0xa2, 0x1d, 0x75, 0x02, 0x14, 0x5d,
    0xc6, 0x78, 0xc8, 0xab, 0xdb, 0xf5, 0xb6, 0x82,
    0x6c, 0xb5, 0x83, 0x87, 0xac, 0x28, 0xb2, 0x55,
    0xb5, 0x9b, 0xc7, 0xc1, 0xb0, 0xb7, 0xf8, 0x4c,
    0xbc, 0x38, 0x0e, 0x8a, 0x04, 0x2a, 0x62, 0x41,
    0x6b, 0xe0, 0x84, 0x09, 0x13, 0xe9, 0xe1, 0xea,
    0xfb, 0xeb, 0x62, 0x71, 0x4b, 0x25, 0xd9, 0x55,
    0x7e, 0x97, 0x57, 0x3b, 0x20, 0x33, 0x96, 0x79,
    0xb5, 0xba, 0x2e, 0x4b, 0x58, 0xae, 0x0b, 0xc8,
    0x60, 0x93, 0x15, 0x55, 0xbe, 0xd8, 0xde, 0x65,
    0x05, 0x6c, 0xb6, 0xc5, 0x66, 0x5d, 0x5e, 0x93,
    0xf7, 0x25, 0x65, 0x98, 0x41, 0x29, 0x86, 0x0c,
    0xf2, 0xf1, 0x14, 0xa2, 0xb3, 0xbd, 0x75, 0x08,
    0x12, 0x83, 0x50, 0xda, 0x1f, 0x23, 0xbe, 0xa3,
    0x1d, 0xf4, 0x9d, 0x1d, 0xb5, 0x84, 0x4e, 0x50,
    0x38, 0x1d, 0x36, 0x48, 0x21, 0x95, 0xd1, 0xac,
    0x81, 0x99, 0x1d, 0xc1, 0x3f, 0x41, 0xe6, 0x9e,
    0x42, 0x5b, 0x0a, 0x48, 0xcc, 0x5f, 0xe0, 0x7d,
    0x3f, 0xc4, 0x6f, 0x0e, 0xfe, 0xc0, 0x2d, 0xfe,
    0x01, 0x2c, 0xd6, 0x9b, 0x5d, 0xbe, 0xba, 0x21,
    0xca, 0x79, 0xcb, 0xe3, 0x49, 0x60, 0xef, 0x68,
    0x05, 0x28, 0x9b, 0x8c, 0xc1, 0x12, 0x3e, 0xdb,
    0xc7, 0x04, 0x7e, 0xa6, 0x74, 0x29, 0xcc, 0x13,
    0xed, 0x07, 0x94, 0x81, 0xd6, 0x96, 0xaa, 0x97,
    0xaa, 0xa5, 0xc0, 0x2f, 0xb5, 0xb5, 0x2e, 0xe6,
    0xfc, 0xca, 0xfa, 0x60, 0x4d, 0x02, 0xf7, 0x19,
    0x9c, 0x5f, 0xa4, 0xe9, 0xf9, 0xf7, 0xf4, 0xc7,
    0x79, 0x9a, 0xc0, 0xb6, 0xcc, 0x58, 0xec, 0xec,
    0xe4, 0x37, 0x22, 0xfa, 0x8b, 0x53,
};

static unsigned char some_data[] = {
    0x2f, 0x2a, 0x0a, 0x20, 0x2a, 0x20, 0x54, 0x68,
    0x69, 0x73, 0x20, 0x66, 0x69, 0x6c, 0x65, 0x20,
    0x69, 0x73, 0x20, 0x66, 0x72, 0x65, 0x65, 0x20,
    0x73, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65,
    0x3b, 0x20, 0x79, 0x6f, 0x75, 0x20, 0x63, 0x61,
    0x6e, 0x20, 0x72, 0x65, 0x64, 0x69, 0x73, 0x74,
    0x72, 0x69, 0x62, 0x75, 0x74, 0x65, 0x20, 0x69,
    0x74, 0x20, 0x61, 0x6e, 0x64, 0x2f, 0x6f, 0x72,
    0x20, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x79, 0x0a,
    0x20, 0x2a, 0x20, 0x69, 0x74, 0x20, 0x75, 0x6e,
    0x64, 0x65, 0x72, 0x20, 0x74, 0x68, 0x65, 0x20,
    0x74, 0x65, 0x72, 0x6d, 0x73, 0x20, 0x6f, 0x66,
    0x20, 0x74, 0x68, 0x65, 0x20, 0x47, 0x4e, 0x55,
    0x20, 0x47, 0x65, 0x6e, 0x65, 0x72, 0x61, 0x6c,
    0x20, 0x50, 0x75, 0x62, 0x6c, 0x69, 0x63, 0x20,
    0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x2c,
    0x20, 0x76, 0x65, 0x72, 0x73, 0x69, 0x6f, 0x6e,
    0x20, 0x32, 0x2c, 0x0a, 0x20, 0x2a, 0x20, 0x61,
    0x73, 0x20, 0x70, 0x75, 0x62, 0x6c, 0x69, 0x73,
    0x68, 0x65, 0x64, 0x20, 0x62, 0x79, 0x20, 0x74,
    0x68, 0x65, 0x20, 0x46, 0x72, 0x65, 0x65, 0x20,
    0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65,
    0x20, 0x46, 0x6f, 0x75, 0x6e, 0x64, 0x61, 0x74,
    0x69, 0x6f, 0x6e, 0x2e, 0x0a, 0x20, 0x2a, 0x0a,
    0x20, 0x2a, 0x20, 0x49, 0x6e, 0x20, 0x61, 0x64,
    0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x74,
    0x6f, 0x20, 0x74, 0x68, 0x65, 0x20, 0x70, 0x65,
    0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e,
    0x73, 0x20, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x65,
    0x20, 0x47, 0x4e, 0x55, 0x20, 0x47, 0x65, 0x6e,
    0x65, 0x72, 0x61, 0x6c, 0x20, 0x50, 0x75, 0x62,
    0x6c, 0x69, 0x63, 0x20, 0x4c, 0x69, 0x63, 0x65,
    0x6e, 0x73, 0x65, 0x2c, 0x0a, 0x20, 0x2a, 0x20,
    0x74, 0x68, 0x65, 0x20, 0x61, 0x75, 0x74, 0x68,
    0x6f, 0x72, 0x73, 0x20, 0x67, 0x69, 0x76, 0x65,
    0x20, 0x79, 0x6f, 0x75, 0x20, 0x75, 0x6e, 0x6c,
    0x69, 0x6d, 0x69, 0x74, 0x65, 0x64, 0x20, 0x70,
    0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f,
    0x6e, 0x20, 0x74, 0x6f, 0x20, 0x6c, 0x69, 0x6e,
    0x6b, 0x20, 0x74, 0x68, 0x65, 0x20, 0x63, 0x6f,
    0x6d, 0x70, 0x69, 0x6c, 0x65, 0x64, 0x0a, 0x20,
    0x2a, 0x20, 0x76, 0x65, 0x72, 0x73, 0x69, 0x6f,
    0x6e, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x69,
    0x73, 0x20, 0x66, 0x69, 0x6c, 0x65, 0x20, 0x69,
    0x6e, 0x74, 0x6f, 0x20, 0x63, 0x6f, 0x6d, 0x62,
    0x69, 0x6e, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x73,
    0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x6f, 0x74,
    0x68, 0x65, 0x72, 0x20, 0x70, 0x72, 0x6f, 0x67,
    0x72, 0x61, 0x6d, 0x73, 0x2c, 0x0a, 0x20, 0x2a,
    0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x6f, 0x20,
    0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75,
    0x74, 0x65, 0x20, 0x74, 0x68, 0x6f, 0x73, 0x65,
    0x20, 0x63, 0x6f, 0x6d, 0x62, 0x69, 0x6e, 0x61,
    0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x77, 0x69,
    0x74, 0x68, 0x6f, 0x75, 0x74, 0x20, 0x61, 0x6e,
    0x79, 0x20, 0x72, 0x65, 0x73, 0x74, 0x72, 0x69,
    0x63, 0x74, 0x69, 0x6f, 0x6e, 0x0a, 0x20, 0x2a,
    0x20, 0x63, 0x6f, 0x6d, 0x69, 0x6e, 0x67, 0x20,
    0x66, 0x72, 0x6f, 0x6d, 0x20, 0x74, 0x68, 0x65,
    0x20, 0x75, 0x73, 0x65, 0x20, 0x6f, 0x66, 0x20,
    0x74, 0x68, 0x69, 0x73, 0x20, 0x66, 0x69, 0x6c,
    0x65, 0x2e, 0x20, 0x20, 0x28, 0x54, 0x68, 0x65,
    0x20, 0x47, 0x65, 0x6e, 0x65, 0x72, 0x61, 0x6c,
    0x20, 0x50, 0x75, 0x62, 0x6c, 0x69, 0x63, 0x20,
    0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x0a,
    0x20, 0x2a, 0x20, 0x72, 0x65, 0x73, 0x74, 0x72,
    0x69, 0x63, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20,
    0x64, 0x6f, 0x20, 0x61, 0x70, 0x70, 0x6c, 0x79,
    0x20, 0x69, 0x6e, 0x20, 0x6f, 0x74, 0x68, 0x65,
    0x72, 0x20, 0x72, 0x65, 0x73, 0x70, 0x65, 0x63,
    0x74, 0x73, 0x3b, 0x20, 0x66, 0x6f, 0x72, 0x20,
    0x65, 0x78, 0x61, 0x6d, 0x70, 0x6c, 0x65, 0x2c,
    0x20, 0x74, 0x68, 0x65, 0x79, 0x20, 0x63, 0x6f,
    0x76, 0x65, 0x72, 0x0a, 0x20, 0x2a, 0x20, 0x6d,
    0x6f, 0x64, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74,
    0x69, 0x6f, 0x6e, 0x20, 0x6f, 0x66, 0x20, 0x74,
    0x68, 0x65, 0x20, 0x66, 0x69, 0x6c, 0x65, 0x2c,
    0x20, 0x61, 0x6e, 0x64, 0x20, 0x64, 0x69, 0x73,
    0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f,
    0x6e, 0x20, 0x77, 0x68, 0x65, 0x6e, 0x20, 0x6e,
    0x6f, 0x74, 0x20, 0x6c, 0x69, 0x6e, 0x6b, 0x65,
    0x64, 0x20, 0x69, 0x6e, 0x74, 0x6f, 0x0a, 0x20,
    0x2a, 0x20, 0x61, 0x20, 0x63, 0x6f, 0x6d, 0x62,
    0x69, 0x6e, 0x65, 0x64, 0x20, 0x65, 0x78, 0x65,
    0x63, 0x75, 0x74, 0x61, 0x62, 0x6c, 0x65, 0x2e,
    0x29, 0x0a, 0x20, 0x2a, 0x0a, 0x20, 0x2a, 0x20,
    0x54, 0x68, 0x69, 0x73, 0x20, 0x66, 0x69, 0x6c,
    0x65, 0x20, 0x69, 0x73, 0x20, 0x64, 0x69, 0x73,
    0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x65, 0x64,
    0x20, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20,
    0x68, 0x6f, 0x70, 0x65, 0x20, 0x74, 0x68, 0x61,
    0x74, 0x20, 0x69, 0x74, 0x20, 0x77, 0x69, 0x6c,
    0x6c, 0x20, 0x62, 0x65, 0x20, 0x75, 0x73, 0x65,
    0x66, 0x75, 0x6c, 0x2c, 0x20, 0x62, 0x75, 0x74,
    0x0a, 0x20, 0x2a, 0x20, 0x57, 0x49, 0x54, 0x48,
    0x4f, 0x55, 0x54, 0x20, 0x41, 0x4e, 0x59, 0x20,
    0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x59,
    0x3b, 0x20, 0x77, 0x69, 0x74, 0x68, 0x6f, 0x75,
    0x74, 0x20, 0x65, 0x76, 0x65, 0x6e, 0x20, 0x74,
    0x68, 0x65, 0x20, 0x69, 0x6d, 0x70, 0x6c, 0x69,
    0x65, 0x64, 0x20, 0x77, 0x61, 0x72, 0x72, 0x61,
    0x6e, 0x74, 0x79, 0x20, 0x6f, 0x66, 0x0a, 0x20,
    0x2a, 0x20, 0x4d, 0x45, 0x52, 0x43, 0x48, 0x41,
    0x4e, 0x54, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54,
    0x59, 0x20, 0x6f, 0x72, 0x20, 0x46, 0x49, 0x54,
    0x4e, 0x45, 0x53, 0x53, 0x20, 0x46, 0x4f, 0x52,
    0x20, 0x41, 0x20, 0x50, 0x41, 0x52, 0x54, 0x49,
    0x43, 0x55, 0x4c, 0x41, 0x52, 0x20, 0x50, 0x55,
    0x52, 0x50, 0x4f, 0x53, 0x45, 0x2e, 0x20, 0x20,
    0x53, 0x65, 0x65, 0x20, 0x74, 0x68, 0x65, 0x20,
    0x47, 0x4e, 0x55, 0x0a, 0x20, 0x2a, 0x20, 0x47,
    0x65, 0x6e, 0x65, 0x72, 0x61, 0x6c, 0x20, 0x50,
    0x75, 0x62, 0x6c, 0x69, 0x63, 0x20, 0x4c, 0x69,
    0x63, 0x65, 0x6e, 0x73, 0x65, 0x20, 0x66, 0x6f,
    0x72, 0x20, 0x6d, 0x6f, 0x72, 0x65, 0x20, 0x64,
    0x65, 0x74, 0x61, 0x69, 0x6c, 0x73, 0x2e, 0x0a,
    0x20, 0x2a, 0x0a, 0x20, 0x2a, 0x20, 0x59, 0x6f,
    0x75, 0x20, 0x73, 0x68, 0x6f, 0x75, 0x6c, 0x64,
    0x20, 0x68, 0x61, 0x76, 0x65, 0x20, 0x72, 0x65,
    0x63, 0x65, 0x69, 0x76, 0x65, 0x64, 0x20, 0x61,
    0x20, 0x63, 0x6f, 0x70, 0x79, 0x20, 0x6f, 0x66,
    0x20, 0x74, 0x68, 0x65, 0x20, 0x47, 0x4e, 0x55,
    0x20, 0x47, 0x65, 0x6e, 0x65, 0x72, 0x61, 0x6c,
    0x20, 0x50, 0x75, 0x62, 0x6c, 0x69, 0x63, 0x20,
    0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x0a,
    0x20, 0x2a, 0x20, 0x61, 0x6c, 0x6f, 0x6e, 0x67,
    0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x74, 0x68,
    0x69, 0x73, 0x20, 0x70, 0x72, 0x6f, 0x67, 0x72,
    0x61, 0x6d, 0x3b, 0x20, 0x73, 0x65, 0x65, 0x20,
    0x74, 0x68, 0x65, 0x20, 0x66, 0x69, 0x6c, 0x65,
    0x20, 0x43, 0x4f, 0x50, 0x59, 0x49, 0x4e, 0x47,
    0x2e, 0x20, 0x20, 0x49, 0x66, 0x20, 0x6e, 0x6f,
    0x74, 0x2c, 0x20, 0x77, 0x72, 0x69, 0x74, 0x65,
    0x20, 0x74, 0x6f, 0x0a, 0x20, 0x2a, 0x20, 0x74,
    0x68, 0x65, 0x20, 0x46, 0x72, 0x65, 0x65, 0x20,
    0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65,
    0x20, 0x46, 0x6f, 0x75, 0x6e, 0x64, 0x61, 0x74,
    0x69, 0x6f, 0x6e, 0x2c, 0x20, 0x35, 0x31, 0x20,
    0x46, 0x72, 0x61, 0x6e, 0x6b, 0x6c, 0x69, 0x6e,
    0x20, 0x53, 0x74, 0x72, 0x65, 0x65, 0x74, 0x2c,
    0x20, 0x46, 0x69, 0x66, 0x74, 0x68, 0x20, 0x46,
    0x6c, 0x6f, 0x6f, 0x72, 0x2c, 0x0a, 0x20, 0x2a,
    0x20, 0x42, 0x6f, 0x73, 0x74, 0x6f, 0x6e, 0x2c,
    0x20, 0x4d, 0x41, 0x20, 0x30, 0x32, 0x31, 0x31,
    0x30, 0x2d, 0x31, 0x33, 0x30, 0x31, 0x2c, 0x20,
    0x55, 0x53, 0x41, 0x2e, 0x0a, 0x20, 0x2a, 0x2f,
    0x0a,
};

static object_data some = {
    some_bytes,
    sizeof(some_bytes),
    "fd8430bc864cfcd5f10e5590f8a447e01b942bfe",
    "blob",
    "test-objects/fd",
    "test-objects/fd/8430bc864cfcd5f10e5590f8a447e01b942bfe",
    some_data,
    sizeof(some_data),
};

static int write_object_data(char *file, void *data, size_t len)
{
    git_file fd;
    int ret;

    if ((fd = gitfo_creat(file, 0664)) < 0)
        return -1;
    ret = gitfo_write(fd, data, len);
    gitfo_close(fd);

    return ret;
}

static int write_object_files(object_data *d)
{
    if (mkdir(odb_dir, 0755) < 0) {
        if (errno == EEXIST) {
            fprintf(stderr, "odb directory \"%s\" already exists!\n", odb_dir);
            return -1;
        }
        fprintf(stderr, "can't make odb directory \"%s\"\n", odb_dir);
        return -1;
    }

    if ((mkdir(d->dir, 0755) < 0) && (errno != EEXIST)) {
        fprintf(stderr, "can't make object directory \"%s\"\n", d->dir);
        return -1;
    }
    if (write_object_data(d->file, d->bytes, d->blen) < 0) {
        fprintf(stderr, "can't write object file \"%s\"\n", d->file);
        return -1;
    }

    return 0;
}

static int remove_object_files(object_data *d)
{
    if (unlink(d->file) < 0) {
        fprintf(stderr, "can't delete object file \"%s\"\n", d->file);
        return -1;
    }
    if ((rmdir(d->dir) < 0) && (errno != ENOTEMPTY)) {
        fprintf(stderr, "can't remove object directory \"%s\"\n", d->dir);
        return -1;
    }

    if (rmdir(odb_dir) < 0) {
        fprintf(stderr, "can't remove odb directory \"%s\"\n", odb_dir);
        return -1;
    }

    return 0;
}

static int cmp_objects(git_obj *o, object_data *d)
{
    if (o->type != git_obj_string_to_type(d->type))
        return -1;
    if (o->len != d->dlen)
        return -1;
    if ((o->len > 0) && (memcmp(o->data, d->data, o->len) != 0))
        return -1;
    return 0;
}

BEGIN_TEST(read_loose_commit)
    git_odb *db;
    git_oid id;
    git_obj obj;

    must_pass(write_object_files(&commit));
    must_pass(git_odb_open(&db, odb_dir));
    must_pass(git_oid_mkstr(&id, commit.id));

    must_pass(git_odb__read_loose(&obj, db, &id));
    must_pass(cmp_objects(&obj, &commit));

    git_obj_close(&obj);
    git_odb_close(db);
    must_pass(remove_object_files(&commit));
END_TEST

BEGIN_TEST(read_loose_tree)
    git_odb *db;
    git_oid id;
    git_obj obj;

    must_pass(write_object_files(&tree));
    must_pass(git_odb_open(&db, odb_dir));
    must_pass(git_oid_mkstr(&id, tree.id));

    must_pass(git_odb__read_loose(&obj, db, &id));
    must_pass(cmp_objects(&obj, &tree));

    git_obj_close(&obj);
    git_odb_close(db);
    must_pass(remove_object_files(&tree));
END_TEST

BEGIN_TEST(read_loose_tag)
    git_odb *db;
    git_oid id;
    git_obj obj;

    must_pass(write_object_files(&tag));
    must_pass(git_odb_open(&db, odb_dir));
    must_pass(git_oid_mkstr(&id, tag.id));

    must_pass(git_odb__read_loose(&obj, db, &id));
    must_pass(cmp_objects(&obj, &tag));

    git_obj_close(&obj);
    git_odb_close(db);
    must_pass(remove_object_files(&tag));
END_TEST

BEGIN_TEST(read_loose_zero)
    git_odb *db;
    git_oid id;
    git_obj obj;

    must_pass(write_object_files(&zero));
    must_pass(git_odb_open(&db, odb_dir));
    must_pass(git_oid_mkstr(&id, zero.id));

    must_pass(git_odb__read_loose(&obj, db, &id));
    must_pass(cmp_objects(&obj, &zero));

    git_obj_close(&obj);
    git_odb_close(db);
    must_pass(remove_object_files(&zero));
END_TEST

BEGIN_TEST(read_loose_one)
    git_odb *db;
    git_oid id;
    git_obj obj;

    must_pass(write_object_files(&one));
    must_pass(git_odb_open(&db, odb_dir));
    must_pass(git_oid_mkstr(&id, one.id));

    must_pass(git_odb__read_loose(&obj, db, &id));
    must_pass(cmp_objects(&obj, &one));

    git_obj_close(&obj);
    git_odb_close(db);
    must_pass(remove_object_files(&one));
END_TEST

BEGIN_TEST(read_loose_two)
    git_odb *db;
    git_oid id;
    git_obj obj;

    must_pass(write_object_files(&two));
    must_pass(git_odb_open(&db, odb_dir));
    must_pass(git_oid_mkstr(&id, two.id));

    must_pass(git_odb__read_loose(&obj, db, &id));
    must_pass(cmp_objects(&obj, &two));

    git_obj_close(&obj);
    git_odb_close(db);
    must_pass(remove_object_files(&two));
END_TEST

BEGIN_TEST(read_loose_some)
    git_odb *db;
    git_oid id;
    git_obj obj;

    must_pass(write_object_files(&some));
    must_pass(git_odb_open(&db, odb_dir));
    must_pass(git_oid_mkstr(&id, some.id));

    must_pass(git_odb__read_loose(&obj, db, &id));
    must_pass(cmp_objects(&obj, &some));

    git_obj_close(&obj);
    git_odb_close(db);
    must_pass(remove_object_files(&some));
END_TEST

