/**CFile****************************************************************

  FileName    [abcUnate.c]

  SystemName  [ABC: Logic synthesis and verification system.]

  PackageName [Network and node package.]

  Synopsis    []

  Author      [Alan Mishchenko]
  
  Affiliation [UC Berkeley]

  Date        [Ver. 1.0. Started - June 20, 2005.]

  Revision    [$Id: abcUnate.c,v 1.00 2005/06/20 00:00:00 alanmi Exp $]

***********************************************************************/

#include "abc.h"

////////////////////////////////////////////////////////////////////////
///                        DECLARATIONS                              ///
////////////////////////////////////////////////////////////////////////

static void Abc_NtkPrintUnateBdd( Abc_Ntk_t * pNtk, int fUseNaive, int fVerbose );
static void Abc_NtkPrintUnateSat( Abc_Ntk_t * pNtk, int fVerbose );

////////////////////////////////////////////////////////////////////////
///                     FUNCTION DEFINITIONS                         ///
////////////////////////////////////////////////////////////////////////

/**Function*************************************************************

  Synopsis    [Detects unate variables of the multi-output function.]

  Description []
               
  SideEffects []

  SeeAlso     []

***********************************************************************/
void Abc_NtkPrintUnate( Abc_Ntk_t * pNtk, int fUseBdds, int fUseNaive, int fVerbose )
{
    if ( fUseBdds || fUseNaive )
        Abc_NtkPrintUnateBdd( pNtk, fUseNaive, fVerbose );
    else
        Abc_NtkPrintUnateSat( pNtk, fVerbose );
}

/**Function*************************************************************

  Synopsis    [Detects unate variables using BDDs.]

  Description []
               
  SideEffects []

  SeeAlso     []

***********************************************************************/
void Abc_NtkPrintUnateBdd( Abc_Ntk_t * pNtk, int fUseNaive, int fVerbose )
{
    Abc_Obj_t * pNode;
    Extra_UnateInfo_t * p;
    DdManager * dd;         // the BDD manager used to hold shared BDDs
    DdNode ** pbGlobal;     // temporary storage for global BDDs
    int TotalSupps = 0;
    int TotalUnate = 0;
    int i, clk = clock();
    int clkBdd, clkUnate;

    // compute the global BDDs
    if ( Abc_NtkGlobalBdds(pNtk, 10000000, 0) == NULL )
        return;
clkBdd = clock() - clk;

    // get information about the network
    dd       = pNtk->pManGlob;
    pbGlobal = (DdNode **)Vec_PtrArray( pNtk->vFuncsGlob );

    // print the size of the BDDs
    printf( "The shared BDD size is %d nodes.\n", Cudd_ReadKeys(dd) - Cudd_ReadDead(dd) );

    // perform naive BDD-based computation
    if ( fUseNaive )
    {
        Abc_NtkForEachCo( pNtk, pNode, i )
        {
            p = Extra_UnateComputeSlow( dd, pbGlobal[i] );
            if ( fVerbose )
                Extra_UnateInfoPrint( p );
            TotalSupps += p->nVars;
            TotalUnate += p->nUnate;
            Extra_UnateInfoDissolve( p );
        }
    }
    // perform smart BDD-based computation
    else 
    {
        // create ZDD variables in the manager
        Cudd_zddVarsFromBddVars( dd, 2 );
        Abc_NtkForEachCo( pNtk, pNode, i )
        {
            p = Extra_UnateComputeFast( dd, pbGlobal[i] );
            if ( fVerbose )
                Extra_UnateInfoPrint( p );
            TotalSupps += p->nVars;
            TotalUnate += p->nUnate;
            Extra_UnateInfoDissolve( p );
        }
    }
clkUnate = clock() - clk - clkBdd;

    // print stats
    printf( "Ins/Outs = %4d/%4d.  Total supp = %5d.  Total unate = %5d.\n",
        Abc_NtkCiNum(pNtk), Abc_NtkCoNum(pNtk), TotalSupps, TotalUnate );
    PRT( "Glob BDDs", clkBdd );
    PRT( "Unateness", clkUnate );
    PRT( "Total    ", clock() - clk );

    // deref the PO functions
    Abc_NtkFreeGlobalBdds( pNtk );
    // stop the global BDD manager
    Extra_StopManager( pNtk->pManGlob );
    pNtk->pManGlob = NULL;
}

/**Function*************************************************************

  Synopsis    [Detects unate variables using SAT.]

  Description []
               
  SideEffects []

  SeeAlso     []

***********************************************************************/
void Abc_NtkPrintUnateSat( Abc_Ntk_t * pNtk, int fVerbose )
{
}

////////////////////////////////////////////////////////////////////////
///                       END OF FILE                                ///
////////////////////////////////////////////////////////////////////////


