#include "connect4_comm.h"

#include <cstring>
#include <mutex>
// #include <glog/logging.h>

#define x first
#define y second

using namespace std;
using namespace Connect4Comm;

Connect4HashValuePair g_hash_weight[BORDER_SIZE_HEIGHT][BORDER_SIZE_LENGTH];

Connect4CoordId g_log2_table[67];
uint64_t g_zobrist_board_hash_weight[4][CONNECT4BOARD_SIZE];
uint64_t g_zobrist_player_hash_weight[4];

namespace Connect4Function {

bool InBoard(const Connect4CoordId id) {
    return 0 <= id && id < CONNECT4BOARD_SIZE;
}

bool InBoard(const Connect4CoordId x, const Connect4CoordId y) {
    return 0 <= x && x < BORDER_SIZE_HEIGHT
        && 0 <= y && y < BORDER_SIZE_LENGTH;
}

bool IsUnset(const Connect4CoordId id) {
    return COORD_UNSET == id;
}

bool IsUnset(const Connect4CoordId x, const Connect4CoordId y) {
    return COORD_UNSET == CoordToId(x, y);
}

bool IsResign(const Connect4CoordId id) {
    return COORD_RESIGN == id;
}

bool IsResign(const Connect4CoordId x, const Connect4CoordId y) {
    return COORD_RESIGN == CoordToId(x, y);
}


void IdToCoord(const Connect4CoordId id, Connect4CoordId &x, Connect4CoordId &y) {
    if (COORD_RESIGN == id) {
        x = y = COORD_RESIGN;
    } else if (!InBoard(id)) {
        x = y = COORD_UNSET;
    } else {
        x = id / BORDER_SIZE_LENGTH;
        y = id % BORDER_SIZE_LENGTH;
    }
}

Connect4CoordId CoordToId(const Connect4CoordId x, const Connect4CoordId y) {
    if (COORD_RESIGN == x && COORD_RESIGN == y) {
        return COORD_RESIGN;
    }
    if (!InBoard(x, y)) {
        return COORD_UNSET;
    }
    return x * BORDER_SIZE_LENGTH + y;
}

void StrToCoord(const string &str, Connect4CoordId &x, Connect4CoordId &y) {
    // CHECK_EQ(str.length(), 2) << "string[" << str << "] length not equal to 2";
    x = str[0] - 'a';
    y = str[1] - 'a';
    if (!InBoard(x, y)) {
        x = y = COORD_UNSET;
    }
}

string CoordToStr(const Connect4CoordId x, const Connect4CoordId y) {
    char buffer[3];
    if (!InBoard(x, y)) {
        buffer[0] = buffer[1] = 'z';
    } else {
        buffer[0] = x + 'a';
        buffer[1] = y + 'a';
    }
    return string(buffer, 2);
}

std::string IdToStr(const Connect4CoordId id) {
    Connect4CoordId x, y;
    IdToCoord(id, x, y);
    return CoordToStr(x, y);
}

Connect4CoordId StrToId(const std::string &str) {
    Connect4CoordId x, y;
    StrToCoord(str, x, y);
    return CoordToId(x, y);
}

once_flag CreateGlobalVariables_once;
void CreateGlobalVariables() {
    call_once(
        CreateGlobalVariables_once,
        []() {
            CreateHashWeights();
            CreateQuickLog2Table();
            CreateZobristHash();
        }
    );
}


void CreateHashWeights() {
    g_hash_weight[0][0] = Connect4HashValuePair(1, 1);
    for (Connect4CoordId i = 1; i < CONNECT4BOARD_SIZE; ++i) {
        g_hash_weight[i / BORDER_SIZE_LENGTH][i % BORDER_SIZE_LENGTH] =
            Connect4HashValuePair(g_hash_weight[(i - 1) / BORDER_SIZE_LENGTH][(i - 1) % BORDER_SIZE_LENGTH].x * g_hash_unit.x,
                            g_hash_weight[(i - 1) / BORDER_SIZE_LENGTH][(i - 1) % BORDER_SIZE_LENGTH].y * g_hash_unit.y);
    }
}

void CreateQuickLog2Table() {
    memset(g_log2_table, -1, sizeof(g_log2_table));
    int tmp = 1;

    for (Connect4CoordId i = 0; i < 64; ++i) {
        g_log2_table[tmp] = i;
        tmp *= 2;
        tmp %= 67;
    }
}

#if defined(_WIN32) || defined(_WIN64)
static int rand_r(unsigned int *seed)
{
    unsigned int next = *seed;
    int result;

    next *= 1103515245;
    next += 12345;
    result = (unsigned int)(next / 65536) % 2048;

    next *= 1103515245;
    next += 12345;
    result <<= 10;
    result ^= (unsigned int)(next / 65536) % 1024;

    next *= 1103515245;
    next += 12345;
    result <<= 10;
    result ^= (unsigned int)(next / 65536) % 1024;

    *seed = next;

    return result;
}
#endif

void CreateZobristHash() {
    uint32_t seed = 0xdeadbeaf;

    for (int i = 0; i < 4; ++i) {
        g_zobrist_player_hash_weight[i] = (uint64_t) rand_r(&seed) << 32 | rand_r(&seed);
        for (int j = 0; j < CONNECT4BOARD_SIZE; ++j) {
            g_zobrist_board_hash_weight[i][j] = (uint64_t) rand_r(&seed) << 32 | rand_r(&seed);
        }
    }

}

} // namespace Connect4Function

#undef y
#undef x
