#include "chinesechess_comm.h"

#include <cstring>
#include <mutex>
// #include <glog/logging.h>

#define x first
#define y second

using namespace std;
using namespace ChineseChessComm;

ChineseChessHashValuePair g_hash_weight[BORDER_SIZE_HEIGHT][BORDER_SIZE_LENGTH];

ChineseChessCoordId g_log2_table[67];
uint64_t g_zobrist_board_hash_weight[4][CHINESECHESSBOARD_SIZE];
uint64_t g_zobrist_player_hash_weight[4];

namespace ChineseChessFunction {

bool InBoard(const ChineseChessCoordId id) {
    return 0 <= id && id < CHINESECHESSBOARD_SIZE;
}

bool InBoard(const ChineseChessCoordId x, const ChineseChessCoordId y) {
    return 0 <= x && x < BORDER_SIZE_HEIGHT
        && 0 <= y && y < BORDER_SIZE_LENGTH;
}

bool IsUnset(const ChineseChessCoordId id) {
    return COORD_UNSET == id;
}

bool IsUnset(const ChineseChessCoordId x, const ChineseChessCoordId y) {
    return COORD_UNSET == CoordToId(x, y);
}

bool IsResign(const ChineseChessCoordId id) {
    return COORD_RESIGN == id;
}

bool IsResign(const ChineseChessCoordId x, const ChineseChessCoordId y) {
    return COORD_RESIGN == CoordToId(x, y);
}


void IdToCoord(const ChineseChessCoordId id, ChineseChessCoordId &x, ChineseChessCoordId &y) {
    if (COORD_RESIGN == id) {
        x = y = COORD_RESIGN;
    } else if (!InBoard(id)) {
        x = y = COORD_UNSET;
    } else {
        x = id / BORDER_SIZE_LENGTH;
        y = id % BORDER_SIZE_LENGTH;
    }
}

ChineseChessCoordId CoordToId(const ChineseChessCoordId x, const ChineseChessCoordId y) {
    if (COORD_RESIGN == x && COORD_RESIGN == y) {
        return COORD_RESIGN;
    }
    if (!InBoard(x, y)) {
        return COORD_UNSET;
    }
    return x * BORDER_SIZE_LENGTH + y;
}

void StrToCoord(const string &str, ChineseChessCoordId &x, ChineseChessCoordId &y) {
    // CHECK_EQ(str.length(), 2) << "string[" << str << "] length not equal to 2";
    x = str[0] - 'a';
    y = str[1] - 'a';
    if (!InBoard(x, y)) {
        x = y = COORD_UNSET;
    }
}

string CoordToStr(const ChineseChessCoordId x, const ChineseChessCoordId y) {
    char buffer[3];
    if (!InBoard(x, y)) {
        buffer[0] = buffer[1] = 'z';
    } else {
        buffer[0] = x + 'a';
        buffer[1] = y + 'a';
    }
    return string(buffer, 2);
}

std::string IdToStr(const ChineseChessCoordId id) {
    ChineseChessCoordId x, y;
    IdToCoord(id, x, y);
    return CoordToStr(x, y);
}

ChineseChessCoordId StrToId(const std::string &str) {
    ChineseChessCoordId x, y;
    StrToCoord(str, x, y);
    return CoordToId(x, y);
}

void ActionToId(const int &action, const ChineseChessStoneColor &stone, const ChineseChessCoordId &from_id, const ChineseChessCoordId &to_id){
    int code[3];
    int i;
    if(action == COORD_RESIGN){
        from_id = to_id = COORD_RESIGN;
        stone = EMPTY;
    }else if(action == COORD_UNSET){
        from_id = to_id = COORD_UNSET;
        stone = EMPTY;
    }else{
        for(i = 0; i < 3; i++){
            code[i] = 0;
        }
        for(i = 0; i < 5; i++){
            code[i] = action % 90;
            action = action / 90;
        }
        stone = code[2];
        from_id = code[1];
        to_id = code[0];
    }
}

void IdToAction(const ChineseChessStoneColor &stone, const ChineseChessCoordId &from_id, const ChineseChessCoordId &to_id, const int &action){
    int code[3];
    int i, j;
    for(i = 0; i < 3; i++){
        code[i] = 1;
    }
    if(from_id == COORD_RESIGN || to_id == COORD_RESIGN){
        action = COORD_RESIGN;
    }else if(!InBoard(from_id) || !InBoard(to_id)){
        action = COORD_UNSET;
    }else if(stone >= 0 && from_id >= 0 && to_id >= 0){
        for(i = 0; i < 3; i++){
            for(j = i; j > 0; j--){
                code[i] *= 90;
            }
        }
        action = stone * code[2] + from_id * code[1] + to_id * code[0];
    }
}

once_flag CreateGlobalVariables_once;
void CreateGlobalVariables() {
    call_once(
        CreateGlobalVariables_once,
        []() {
            CreateHashWeights();
            CreateQuickLog2Table();
            CreateZobristHash();
        }
    );
}


void CreateHashWeights() {
    g_hash_weight[0][0] = ChineseChessHashValuePair(1, 1);
    for (ChineseChessCoordId i = 1; i < CHINESECHESSBOARD_SIZE; ++i) {
        g_hash_weight[i / BORDER_SIZE_LENGTH][i % BORDER_SIZE_LENGTH] =
            ChineseChessHashValuePair(g_hash_weight[(i - 1) / BORDER_SIZE_LENGTH][(i - 1) % BORDER_SIZE_LENGTH].x * g_hash_unit.x,
                            g_hash_weight[(i - 1) / BORDER_SIZE_LENGTH][(i - 1) % BORDER_SIZE_LENGTH].y * g_hash_unit.y);
    }
}

void CreateQuickLog2Table() {
    memset(g_log2_table, -1, sizeof(g_log2_table));
    int tmp = 1;

    for (ChineseChessCoordId i = 0; i < 64; ++i) {
        g_log2_table[tmp] = i;
        tmp *= 2;
        tmp %= 67;
    }
}

#if defined(_WIN32) || defined(_WIN64)
static int rand_r(unsigned int *seed)
{
    unsigned int next = *seed;
    int result;

    next *= 1103515245;
    next += 12345;
    result = (unsigned int)(next / 65536) % 2048;

    next *= 1103515245;
    next += 12345;
    result <<= 10;
    result ^= (unsigned int)(next / 65536) % 1024;

    next *= 1103515245;
    next += 12345;
    result <<= 10;
    result ^= (unsigned int)(next / 65536) % 1024;

    *seed = next;

    return result;
}
#endif

void CreateZobristHash() {
    uint32_t seed = 0xdeadbeaf;

    for (int i = 0; i < 4; ++i) {
        g_zobrist_player_hash_weight[i] = (uint64_t) rand_r(&seed) << 32 | rand_r(&seed);
        for (int j = 0; j < CHINESECHESSBOARD_SIZE; ++j) {
            g_zobrist_board_hash_weight[i][j] = (uint64_t) rand_r(&seed) << 32 | rand_r(&seed);
        }
    }

}

} // namespace ChineseChessFunction

#undef y
#undef x
