#include "chess_env.h"
 
#include <algorithm>
#include <cmath>

using namespace std;
using namespace ChessComm;
using namespace ChessFunction;
using namespace ChessFeature;

ChessEnv::ChessEnv() {
    CreateGlobalVariables();

    FOR_EACHCOORD(i) {
        stones_[i].self_id = i;
    }

    action_list_.clear();
    feature_history_list_.clear();
    memset(board_state_, 0, sizeof(board_state_));
    memset(move_count_, 0, sizeof(move_count_));
    current_player_ = WHITE;
    last_action_ = COORD_UNSET;
    is_resign_ = false;
    state_ = 0;
    material_ = 0;
    board_hash_states_.clear();
    zobrist_hash_value_ = 0;
}

//ChessEnv::ChessEnv(const ChessEnv &ge) : ChessEnv() {
//     CopyFrom(ge);
// }
 
ChessEnv::~ChessEnv() {

}

 // void ChessEnv::CopyFrom(const ChessEnv &src) {
 //     *this = src;
 // }


void ChessEnv::GetSensibleMove() {
    // Add new feature plane
    string plane;
    // white
    {
        plane = "";
        FOR_EACHCOORD(id) {
            if (board_state_[id] == WHITE) {
                plane += "1";
            } else {
                plane += "0";
            }
        }
        feature_history_list_.push_back(plane);
    }
    // black
    {
        plane = "";
        FOR_EACHCOORD(id) {
            if (board_state_[id] == BLACK) {
                plane += "1";
            } else {
                plane += "0";
            }
        }
        feature_history_list_.push_back(plane);
    }

    Find_Legal_Moves();
}
 
int ChessState::Move(int action) {
    ChessCoordId from_id, to_id;
    ChessStoneColor captured_stone, promoted_stone, piece;
    Find_Legal_Moves();
    if (!IsLegal(action)) {
        return -1;
    }
    if(IsResign(action)){
      is_resign_ = true;
      return 0;
    }

    last_action_ = action;

    {
        zobrist_hash_value_ ^= g_zobrist_player_hash_weight[Self()];
        zobrist_hash_value_ ^= g_zobrist_player_hash_weight[Opponent()];
        zobrist_hash_value_ ^= g_zobrist_board_hash_weight[Self()][action];
    }
    ChessFunction::ActionToId(action, piece, captured_stone, promoted_stone, from_id, to_id);

    ++move_count_[action];
    switch (captured_stone)
    {
        case ChessComm::WHITE_PAWN : case ChessComm::BLACK_PAWN : material_ -= 1; break;
        case ChessComm::WHITE_KNIGHT : case ChessComm::BLACK_KNIGHT : material_ -= 3; break;
        case ChessComm::WHITE_BISHOP : case ChessComm::BLACK_BISHOP : material_ -= 3; break;
        case ChessComm::WHITE_ROOK : case ChessComm::BLACK_ROOK : material_ -= 5; break;
        case ChessComm::WHITE_QUEEN : case ChessComm::BLACK_QUEEN : material_ -= 9; break;
        default : break;
    }

    switch (promoted_stone)
    {
        case ChessComm::WHITE_KNIGHT : case ChessComm::BLACK_KNIGHT : material_ -= 3 - 1; break;
        case ChessComm::WHITE_BISHOP : case ChessComm::BLACK_BISHOP : material_ -= 3 - 1; break;
        case ChessComm::WHITE_ROOK : case ChessComm::BLACK_ROOK : material_ -= 5 - 1; break;
        case ChessComm::WHITE_QUEEN : case ChessComm::BLACK_QUEEN : material_ -= 9 - 1; break;
        default : break;
    }
    board_state_[to_id] = board_state_[from_id];
    if (promoted_stone != ChessComm::EMPTY){
        board_state_[to_id] = promoted_stone;
    }
    board_state_[from_id] = ChessComm::EMPTY;

    HandOff();
    material_ = - material_;
    GetSensibleMove();


    return 0;
}

bool ChessEnv::IsSquareThreatened(ChessCoordId& id) const{  
    if(ChessFunction::InBoard(id)){
        if (current_player_ == ChessComm::WHITE){
            // Check for pawns
            if (board_state_[id + ChessComm::NW] == ChessComm::BLACK_PAWN || board_state_[id + ChessComm::NE] == ChessComm::BLACK_PAWN){
                return true;
            }
            // Check for knigts
            {
                int dirs[8] = {ChessComm::NNW, ChessComm::NNE, ChessComm::NWW, ChessComm::NEE, ChessComm::SSW, ChessComm::SSE, ChessComm::SWW, ChessComm::SEE};

                for (int k = 0; k < 8; ++k){
                    int dir = dirs[k];
                    if (board_state_[id + dir] == ChessComm::BLACK_KNIGHT){
                        return true;
                    }
                }
            }

            // Check for diagonal (bishop and queen)
            {
                int dirs[4] = {ChessComm::NW, ChessComm::NE, ChessComm::SW, ChessComm::SE};

                for (int k = 0; k < 4; ++k){
                    int dir = dirs[k];
                    int id2 = id;
                    while (1){
                        id2 += dir;
                        if (board_state_[id2] == ChessComm::BLACK_BISHOP || board_state_[id2] == ChessComm::BLACK_QUEEN){
                            return true;
                        }
                        if (board_state_[id2] != ChessComm::EMPTY){
                            break;
                        }
                    }

                }
            }

            // Check for horizontal/vertical (rook and queen)
            {
                int dirs[4] = {ChessComm::N, ChessComm::S, ChessComm::W, ChessComm::E};

                for (int k = 0; k < 4; ++k)
                {
                    int dir = dirs[k];
                    int id2 = id;
                    while (1){
                        id2 += dir;
                        if (board_state_[id2] == ChessComm::BLACK_ROOK || board_state_[id2] == ChessComm::BLACK_QUEEN){
                            return true;
                        }
                        if (board_state_[id2] != ChessComm::EMPTY){
                            break;
                        }
                    }

                }
            }

            // Check for king
            {
                int dirs[8] = {ChessComm::NW, ChessComm::NE, ChessComm::SW, ChessComm::SE, ChessComm::N, ChessComm::S, ChessComm::W, ChessComm::E};

                for (int k = 0; k < 8; ++k){
                    int dir = dirs[k];
                    if (board_state_[id + dir] == ChessComm::BLACK_KING){
                        return true;
                    }
                }
            }
        }else{
            // Check for pawns
            if (board_state_[id + ChessComm::SW] == ChessComm::WHITE_PAWN || board_state_[id + ChessComm::SE] == ChessComm::WHITE_PAWN){
                return true;
            }

            // Check for knigts
            {
                int dirs[8] = {ChessComm::NNW, ChessComm::NNE, ChessComm::NWW, ChessComm::NEE, ChessComm::SSW, ChessComm::SSE, ChessComm::SWW, ChessComm::SEE};

                for (int k = 0; k < 8; ++k){
                    int dir = dirs[k];
                    if (board_state_[id + dir] == ChessComm::WHITE_KNIGHT){
                        return true;
                    }
                }
            }

            // Check for diagonal (bishop and queen)
            {
                int dirs[4] = {ChessComm::NW, ChessComm::NE, ChessComm::SW, ChessComm::SE};

                for (int k = 0; k < 4; ++k){
                    int dir = dirs[k];
                    int id2 = id;
                    while (1){
                        id2 += dir;
                        if (board_state_[id2] == ChessComm::WHITE_BISHOP || board_state_[id2] == ChessComm::WHITE_QUEEN){
                            return true;
                        }
                        if (board_state_[id2] != ChessComm::EMPTY){
                            break;
                        }
                    }

                }
            }

            // Check for horizontal/vertical (rook and queen)
            {
                int dirs[4] = {ChessComm::N, ChessComm::S, ChessComm::W, ChessComm::E};

                for (int k = 0; k < 4; ++k){
                    int dir = dirs[k];
                    int id2 = id;
                    while (1){
                        id2 += dir;
                        if (board_state_[id2] == ChessComm::WHITE_ROOK || board_state_[id2] == ChessComm::WHITE_QUEEN){
                            return true;
                        }
                        if (board_state_[id2] != ChessComm::EMPTY){
                            break;
                        }
                    }

                }
            }

            // Check for king
            {
                int dirs[8] = {ChessComm::NW, ChessComm::NE, ChessComm::SW, ChessComm::SE,ChessComm:: N, ChessComm::S, ChessComm::W, ChessComm::E};

                for (int k = 0; k < 8; ++k){
                    int dir = dirs[k];
                    if (board_state_[id + dir] == ChessComm::WHITE_KING){
                        return true;
                    }
                }
            }
        }
    }

    return false;
}

bool ChessEnv::IsKingInCheck() const{
    ChessStoneColor king;
    ChessCoordId kingid = ChessComm::COORD_UNSET;

    if (current_player_ == ChessComm::WHITE){
        king = ChessComm::WHITE_KING;
    }
    else if(current_player_ == ChessComm::BLACK){
        king = ChessComm::BLACK_KING;
    }

    // Look for our king
    FOR_EACHCOORD(id){
        if(board_state_[id] == king){
            kingid = id;
            break;
        }
    }

    return IsSquareThreatened(kingid);

} // end of IsKingInCheck

void ChessEnv::Find_Legal_Moves() const{    
    action_list_.clear();
    int action;

    if (current_player_ == ChessComm::WHITE){
        for (int i = 0; i < ChessComm::CHESSBOARD_SIZE; ++i){
            ChessStoneColor piece = board_state_[i];
            int j;

            switch (piece){
                case ChessComm::WHITE_PAWN : // White pawn
                    //j = i + 10; // One square forward
                    j = i - ChessComm::BORDER_SIZE; // One square forward
                    if (m_board[j] == EM){
                        //if (i > 80) // Check for promotion
                        if (i < 2 * ChessComm::BORDER_SIZE) // Check for promotion
                        {
                            {
                                ChessFunction::IdToAction(piece, ChessComm::EMPTY, ChessComm::WHITE_QUEEN, i, j, action);
                                //CMove move(piece, i, j, EM, WQ);
                                action_list_.push_back(action);
                            }
                            {
                                ChessFunction::IdToAction(piece, ChessComm::EMPTY, ChessComm::WHITE_ROOK, i, j, action);
                                //CMove move(piece, i, j, EM, WR);
                                action_list_.push_back(action);
                            }
                            {
                                ChessFunction::IdToAction(piece, ChessComm::EMPTY, ChessComm::WHITE_BISHOP, i, j, action);
                                //CMove move(piece, i, j, EM, WB);
                                action_list_.push_back(action);
                            }
                            {
                                ChessFunction::IdToAction(piece, ChessComm::EMPTY, ChessComm::WHITE_KNIGHT, i, j, action);
                                //CMove move(piece, i, j, EM, WN);
                                action_list_.push_back(action);
                            }
                        }
                        else // regular pawn move
                        {
                            ChessFunction::IdToAction(piece, ChessComm::EMPTY, ChessComm::EMPTY, i, j, action);
                            //CMove move(piece, i, j, EM);
                            action_list_.push_back(action);
                        }

                        //j = i + 20; // Two squares forward
                        j = i - 2 * ChessComm::BORDER_SIZE;
                        if (board_state_[j] == ChessComm::EMPTY){
                            //if (i < 40) // Only from second rank
                            if(i >= 6 * ChessComm::BORDER_SIZE)
                            {
                                ChessFunction::IdToAction(piece, ChessComm::EMPTY, ChessComm::EMPTY, i, j, action);
                                //CMove move(piece, i, j, EM);
                                action_list_.push_back(action);
                            }
                        }
                    }

                    //j = i + 9; // Diagonal capture
                    j = i - ChessComm::BORDER_SIZE - 1;
                    //if ((m_board[j] != IV && m_board[j] < 0))
                    if ((ChessFunction::InBoard(j) && current_player_ == ChessComm::BLACK))
                    {
                        //if (i > 80) // Check for promotion
                        if (i < 2 * ChessComm::BORDER_SIZE) // Check for promotion
                        {
                            {
                                ChessFunction::IdToAction(piece, board_state_[j], ChessComm::WHITE_QUEEN, i, j, action);
                                //CMove move(piece, i, j, m_board[j], WQ);
                                action_list_.push_back(action);
                            }
                            {
                                ChessFunction::IdToAction(piece, board_state_[j], ChessComm::WHITE_ROOK, i, j, action);
                                //CMove move(piece, i, j, m_board[j], WR);
                                action_list_.push_back(action);
                            }
                            {
                                ChessFunction::IdToAction(piece, board_state_[j], ChessComm::WHITE_BISHOP, i, j, action);
                                //CMove move(piece, i, j, m_board[j], WB);
                                action_list_.push_back(action);
                            }
                            {
                                ChessFunction::IdToAction(piece, board_state_[j], ChessComm::WHITE_KNIGHT, i, j, action);
                                //CMove move(piece, i, j, m_board[j], WN);
                                action_list_.push_back(action);
                            }
                        }
                        else
                        {
                            ChessFunction::IdToAction(piece, board_state_[j], ChessComm::EMPTY, i, j, action);
                            //CMove move(piece, i, j, m_board[j]);
                            action_list_.push_back(action);
                        }
                    }

                    //j = i + 11; // Diagonal capture
                    j = i - ChessComm::BORDER_SIZE + 1;
                    //if ((m_board[j] != IV && m_board[j] < 0))
                    if((ChessFunction::InBoard(j) && current_player_ == ChessComm::BLACK))
                    {
                        //if (i > 80) // Check for promotion
                        if(i < 2 * ChessComm::BORDER_SIZE)
                        {
                            {
                                ChessFunction::IdToAction(piece, board_state_[j], ChessComm::WHITE_QUEEN, i, j, action);
                                //CMove move(piece, i, j, m_board[j], WQ);
                                action_list_.push_back(action);
                            }
                            {
                                ChessFunction::IdToAction(piece, board_state_[j], ChessComm::WHITE_ROOK, i, j, action);
                                //CMove move(piece, i, j, m_board[j], WR);
                                action_list_.push_back(action);
                            }
                            {
                                ChessFunction::IdToAction(piece, board_state_[j], ChessComm::WHITE_BISHOP, i, j, action);
                                //CMove move(piece, i, j, m_board[j], WB);
                                action_list_.push_back(action);
                            }
                            {
                                ChessFunction::IdToAction(piece, board_state_[j], ChessComm::WHITE_KNIGHT, i, j, action);
                                //CMove move(piece, i, j, m_board[j], WN);
                                action_list_.push_back(action);
                            }
                        }
                        else
                        {
                            ChessFunction::IdToAction(piece, board_state_[j], ChessComm::EMPTY, i, j, action);
                            //CMove move(piece, i, j, m_board[j]);
                            action_list_.push_back(action);
                        }
                    }
                    break;

                case ChessComm::WHITE_KNIGHT : // White knight 
                    {
                        int dirs[8] = {ChessComm::NNW, ChessComm::NNE, ChessComm::NWW, ChessComm::NEE, ChessComm::SSW, ChessComm::SSE, ChessComm::SWW, ChessComm::SEE};

                        for (int k = 0; k < 8; ++k)
                        {
                            int dir = dirs[k];

                            j = i + dir;
                            //if (m_board[j] != IV) 
                            if (ChessFunction::InBoard(j)){
                                //if (m_board[j] <= 0)
                                if(current_player_ == ChessComm::BLACK || board_state_[j] == ChessComm::EMPTY)
                                {
                                    ChessFunction::IdToAction(piece, board_state_[j], ChessComm::EMPTY, i, j, action);
                                    //CMove move(piece, i, j, m_board[j]);
                                    action_list_.push_back(action);
                                }
                            }
                        }
                    }

                    break;

                case ChessComm::WHITE_BISHOP : // White bishop
                    {
                        int dirs[4] = {ChessComm::NE, ChessComm::NW, ChessComm::SE, ChessComm::SW};

                        for (int k = 0; k < 4; ++k)
                        {
                            int dir = dirs[k];
                            j = i;
                            while (1)
                            {
                                j += dir;
                                //if (m_board[j] == IV) break;
                                if(!InBoard(j)){
                                    break;
                                }
                                //if (m_board[j] <= 0)
                                if(current_player_ == ChessComm::BLACK || board_state_[j] == ChessComm::EMPTY)
                                {
                                    ChessFunction::IdToAction(piece, board_state_[j], ChessComm::EMPTY, i, j, action);
                                    //CMove move(piece, i, j, m_board[j]);
                                    action_list_.push_back(action);
                                    //if (m_board[j] < 0)
                                    if(current_player_ == ChessComm::BLACK){
                                        break;
                                    }
                                }
                                else{
                                    break;
                                }
                            }
                        }
                    }

                    break;

                case ChessComm::WHITE_ROOK : // White rook
                    {
                        int dirs[4] = {ChessComm::N, ChessComm::E, ChessComm::S, ChessComm::W};

                        for (int k = 0; k < 4; ++k)
                        {
                            int dir = dirs[k];
                            j = i;
                            while (1)
                            {
                                j += dir;
                                //if (m_board[j] == IV) break;
                                if(!InBoard(j)){
                                    break;
                                }
                                //if (m_board[j] <= 0)
                                if(current_player_ == ChessComm::BLACK || board_state_[j] == ChessComm::EMPTY)
                                {
                                    ChessFunction::IdToAction(piece, board_state_[j], ChessComm::EMPTY, i, j, action);
                                    //CMove move(piece, i, j, m_board[j]);
                                    action_list_.push_back(action);
                                    //if (m_board[j] < 0)
                                    if(current_player_ == ChessComm::BLACK){
                                        break;
                                    }
                                }
                                else{
                                    break;
                                }
                            }
                        }
                    }

                    break;

                case ChessComm::WHITE_QUEEN : // White queen
                    {
                        int dirs[8] = {ChessComm::NE, ChessComm::NW, ChessComm::SE, ChessComm::SW, ChessComm::N, ChessComm::E, ChessComm::S, ChessComm::W};

                        for (int k = 0; k < 8; ++k)
                        {
                            int dir = dirs[k];
                            j = i;
                            while (1)
                            {
                                j += dir;
                                //if (m_board[j] == IV) break;
                                if(!InBoard(j)){
                                    break;
                                }
                                //if (m_board[j] <= 0)
                                if(current_player_ == ChessComm::BLACK || board_state_[j] == ChessComm::EMPTY)
                                {
                                    ChessFunction::IdToAction(piece, board_state_[j], ChessComm::EMPTY, i, j, action);
                                    //CMove move(piece, i, j, m_board[j]);
                                    action_list_.push_back(action);
                                    //if (m_board[j] < 0)
                                    if(current_player_ == ChessComm::BLACK){
                                        break;
                                    }
                                }
                                else{
                                    break;
                                }
                            }
                        }
                    }

                    break;

                case ChessComm::WHITE_KING : // White king
                    {
                        int dirs[8] = {ChessComm::NE, ChessComm::NW, ChessComm::SE, ChessComm::SW, ChessComm::N, ChessComm::E, ChessComm::S, ChessComm::W};

                        for (int k = 0; k < 8; ++k)
                        {
                            int dir = dirs[k];

                            j = i + dir;
                            //if (m_board[j] != IV) 
                            if(ChessFunction::InBoard(j)){
                                //if (m_board[j] <= 0)
                                if(current_player_ == ChessComm::BLACK || board_state_[j] == ChessComm::EMPTY)
                                {
                                    ChessFunction::IdToAction(piece, board_state_[j], ChessComm::EMPTY, i, j, action);
                                    //CMove move(piece, i, j, m_board[j]);
                                    action_list_.push_back(action);
                                }
                            }
                        }

                    }

                    break;

                default : // Invalid, wrong color, or empty
                    continue;
            }

        }
    }

    else{
        //for (int i=A1; i<=H8; ++i)
        for(int i = 0; i < ChessComm::CHESSBOARD_SIZE; i++)
        {
            ChessStoneColor piece = board_state_[i];
            int j;

            switch (piece)
            {
                case ChessComm::BLACK_PAWN : // Black pawn
                    //j = i - 10; // One squre forward
                    j = i + ChessComm::BORDER_SIZE; // One squre forward
                    //if (m_board[j] == EM)
                    if(board_state_[j] == ChessComm::EMPTY)
                    {
                        //if (i < 40) // Check for promotion
                        if(i >= 6 * ChessComm::BORDER_SIZE)
                        {
                            {
                                ChessFunction::IdToAction(piece, ChessComm::EMPTY, ChessComm::BLACK_QUEEN, i, j, action);
                                //CMove move(piece, i, j, EM, BQ);
                                action_list_.push_back(action);
                            }
                            {
                                ChessFunction::IdToAction(piece, ChessComm::EMPTY, ChessComm::BLACK_ROOK, i, j, action);
                                //CMove move(piece, i, j, EM, BR);
                                action_list_.push_back(action);
                            }
                            {
                                ChessFunction::IdToAction(piece, ChessComm::EMPTY, ChessComm::BLACK_BISHOP, i, j, action);
                                //CMove move(piece, i, j, EM, BB);
                                action_list_.push_back(action);
                            }
                            {
                                ChessFunction::IdToAction(piece, ChessComm::EMPTY, ChessComm::BLACK_KNIGHT, i, j, action);
                                //CMove move(piece, i, j, EM, BN);
                                action_list_.push_back(action);
                            }
                        }
                        else // Regular pawn move
                        {
                            ChessFunction::IdToAction(piece, ChessComm::EMPTY, ChessComm::EMPTY, i, j, action);
                            //CMove move(piece, i, j, EM);
                            action_list_.push_back(action);
                        }

                        //j = i - 20; // Two squares forward
                        j = i + 2 * ChessComm::BORDER_SIZE; // Two squares forward
                        //if (m_board[j] == EM)
                        if(board_state_[j] == ChessComm::EMPTY)
                        {
                            //if (i > 80) // Only from seventh rank
                            if (i < 2 * ChessComm::BORDER_SIZE) // Only from seventh rank
                            {
                                ChessFunction::IdToAction(piece, ChessComm::EMPTY, ChessComm::EMPTY, i, j, action);
                                //CMove move(piece, i, j, EM);
                                action_list_.push_back(action);
                            }
                        }
                    }

                    //j = i - 9; // Diagonal capture
                    j = i + ChessComm::BORDER_SIZE + 1; // Diagonal capture
                    //if ((m_board[j] != IV && m_board[j] > 0))
                    if(ChessFunction::InBoard(j) && current_player_ == ChessComm::WHITE)
                    {
                        //if (i < 40) // Check for promotion
                        if (i >= 6 * ChessComm::BORDER_SIZE) // Check for promotion
                        {
                            {
                                ChessFunction::IdToAction(piece, board_state_[j], ChessComm::BLACK_QUEEN, i, j, action);
                                //CMove move(piece, i, j, m_board[j], BQ);
                                action_list_.push_back(action);
                            }
                            {
                                ChessFunction::IdToAction(piece, board_state_[j], ChessComm::BLACK_ROOK, i, j, action);
                                //CMove move(piece, i, j, m_board[j], BR);
                                action_list_.push_back(action);
                            }
                            {
                                ChessFunction::IdToAction(piece, board_state_[j], ChessComm::BLACK_BISHOP, i, j, action);
                                //CMove move(piece, i, j, m_board[j], BB);
                                action_list_.push_back(action);
                            }
                            {
                                ChessFunction::IdToAction(piece, board_state_[j], ChessComm::BLACK_KNIGHT, i, j, action);
                                //CMove move(piece, i, j, m_board[j], BN);
                                action_list_.push_back(action);
                            }
                        }
                        else
                        {
                            ChessFunction::IdToAction(piece, board_state_[j], ChessComm::EMPTY, i, j, action);
                            //CMove move(piece, i, j, m_board[j]);
                            action_list_.push_back(action);
                        }
                    }

                    //j = i - 11; // Diagonal capture
                    j = i + ChessComm::BORDER_SIZE - 1;
                    //if ((m_board[j] != IV && m_board[j] > 0))
                    if(ChessComm::InBoard[j] && current_player_ == ChessComm::WHITE)
                    {
                        //if (i < 40) // Check for promotion
                        if(i >= 6 * ChessComm::BORDER_SIZE)
                        {
                            {
                                ChessFunction::IdToAction(piece, board_state_[j], ChessComm::BLACK_QUEEN, i, j, action);
                                //CMove move(piece, i, j, m_board[j], BQ);
                                action_list_.push_back(action);
                            }
                            {
                                ChessFunction::IdToAction(piece, board_state_[j], ChessComm::BLACK_ROOK, i, j, action);
                                //CMove move(piece, i, j, m_board[j], BR);
                                action_list_.push_back(action);
                            }
                            {
                                ChessFunction::IdToAction(piece, board_state_[j], ChessComm::BLACK_BISHOP, i, j, action);
                                //CMove move(piece, i, j, m_board[j], BB);
                                action_list_.push_back(action);
                            }
                            {
                                ChessFunction::IdToAction(piece, board_state_[j], ChessComm::BLACK_KNIGHT, i, j, action);
                                //CMove move(piece, i, j, m_board[j], BN);
                                action_list_.push_back(action);
                            }
                        }
                        else
                        {
                            ChessFunction::IdToAction(piece, board_state_[j], ChessComm::EMPTY, i, j, action);
                            //CMove move(piece, i, j, m_board[j]);
                            action_list_.push_back(action);
                        }
                    }
                    break;

                case ChessComm::BLACK_KNIGHT : // Black knight
                    {
                        int dirs[8] = {ChessComm::NNW, ChessComm::NNE, ChessComm::NWW, ChessComm::NEE, ChessComm::SSW, ChessComm::SSE, ChessComm::SWW, ChessComm::SEE};

                        for (int k = 0; k < 8; ++k)
                        {
                            int dir = dirs[k];

                            j = i + dir;
                            //if (m_board[j] != IV) 
                            if(ChessFunction::InBoard(j)){
                                //if (m_board[j] >= 0)
                                if(current_player_ == ChessComm::WHITE || board_state_[j] == ChessComm::EMPTY)
                                {
                                    ChessFunction::IdToAction(piece, board_state_[j], ChessComm::EMPTY, i, j, action);
                                    //CMove move(piece, i, j, m_board[j]);
                                    action_list_.push_back(action);
                                }
                            }
                        }
                    }

                    break;

                case ChessComm::BLACK_BISHOP : // Black bishop
                    {
                        int dirs[4] = {ChessComm::NE, ChessComm::NW, ChessComm::SE, ChessComm::SW};

                        for (int k = 0; k < 4; ++k)
                        {
                            int dir = dirs[k];
                            j = i;
                            while (1)
                            {
                                j += dir;
                                //if (m_board[j] == IV) break;
                                if(!InBoard(j)){
                                    break;
                                }
                                //if (m_board[j] >= 0)
                                if(current_player_ == ChessComm::WHITE || board_state_[j] == ChessComm::EMPTY)
                                {
                                    ChessFunction::IdToAction(piece, board_state_[j], ChessComm::EMPTY, i, j, action);
                                    //CMove move(piece, i, j, m_board[j]);
                                    action_list_.push_back(action);
                                    //if (m_board[j] > 0)
                                    if(current_player_ == ChessComm::White){
                                        break;
                                    }
                                }
                                else{
                                    break;
                                }
                            }
                        }
                    }

                    break;

                case ChessComm::BLACK_ROOK : // Black rook
                    {
                        int dirs[4] = {ChessComm::N, ChessComm::E, ChessComm::S, ChessComm::W};

                        for (int k = 0; k < 4; ++k)
                        {
                            int dir = dirs[k];
                            j = i;
                            while (1)
                            {
                                j += dir;
                                //if (m_board[j] == IV) break;
                                if(!InBoard(j)){
                                    break;
                                }
                                //if (m_board[j] >= 0)
                                if(current_player_ == ChessComm::WHITE || board_state_[j] == ChessComm:EMPTY)
                                {
                                    ChessFunction::IdToAction(piece, board_state_[j], ChessComm::EMPTY, i, j, action);
                                    //CMove move(piece, i, j, m_board[j]);
                                    action_list_.push_back(action);
                                    //if (m_board[j] > 0)
                                    if(current_player_ == ChessComm::WHITE){
                                        break;
                                    }
                                }
                                else{
                                    break;
                                }
                            }
                        }
                    }

                    break;

                case ChessComm::BLACK_QUEEN : // Black queen
                    {
                        int dirs[8] = {ChessComm::NE, ChessComm::NW, ChessComm::SE, ChessComm::SW, ChessComm::N, ChessComm::E, ChessComm::S, ChessComm::W};

                        for (int k = 0; k < 8; ++k)
                        {
                            int dir = dirs[k];
                            j = i;
                            while (1)
                            {
                                j += dir;
                                //if (m_board[j] == IV) break;
                                if(!InBoard(j)){
                                    break;
                                }
                                //if (m_board[j] >= 0)
                                if(current_player_ == ChessComm::WHITE || board_state_[j] == ChessComm::EMPTY)
                                {
                                    ChessFunction::IdToAction(piece, board_state_[j], ChessComm::EMPTY, i, j, action);
                                    //CMove move(piece, i, j, m_board[j]);
                                    action_list_.push_back(action);
                                    //if (m_board[j] > 0)
                                    if(current_player_ == ChessComm::WHITE){
                                        break;
                                    }
                                }
                                else{
                                    break;
                                }
                            }
                        }
                    }

                    break;

                case ChessComm::BLACK_KING: // Black king
                    {
                        int dirs[8] = {ChessComm::NE, ChessComm::NW, ChessComm::SE, ChessComm::SW, ChessComm::N, ChessComm::E, ChessComm::S, ChessComm::W};

                        for (int k = 0; k < 8; ++k)
                        {
                            int dir = dirs[k];

                            j = i + dir;
                            //if (m_board[j] != IV) 
                            if(ChessFunction::InBoard(j)){
                                //if (m_board[j] >= 0)
                                if(current_player_ == ChessComm::WHITE || board_state_[j] == ChessComm::EMPTY)
                                {
                                    ChessFunction::IdToAction(piece, board_state_[j], ChessComm::EMPTY, i, j, action);
                                    //CMove move(piece, i, j, m_board[j]);
                                    action_list_.push_back(action);
                                }
                            }
                        }

                    }

                    break;

                default : // Invalid, wrong color, or empty
                    continue;
            }

        }
    }
}

bool ChessEnv::IsMoveValid(int &action) const{
    ChessCoordId from_id, to_id;
    ChessStoneColor captured_stone, promoted_stone, piece;
    for (unsigned int i=0; i < action_list_.size(); ++i){
        if (action_list_[i] == action){
            ChessFunction::ActionToId(action, piece, captured_stone, promoted_stone, from_id, to_id);
            piece = board_state_[from_id];
            captured_stone = board_state_[to_id];
            ChessFunction::IdToAction(piece, captured_stone, promoted_stone, from_id, to_id, action);
            return true;
        }
    }
    return false;
}



int ChessEnv::CalcResult() const {
    ChessStoneColor king;
    ChessCoordId kingid = ChessComm::COORD_UNSET;

    if (current_player_ == ChessComm::WHITE){
        king = ChessComm::WHITE_KING;
    }
    else if(current_player_ == ChessComm::BLACK){
        king = ChessComm::BLACK_KING;
    }

    FOR_EACHCOORD(id){
        if(board_state_[id] == king){
            kingid = id;
            break;
        }
    }
   
    if(kingid == ChessComm::COORD_UNSET){
        if(current_player_ == ChessComm::WHITE){
            state_ = 2;//BLACK WIN!
        }else if (current_player_ == ChessComm::BLACK){
            state_ = 1;//WHITE WIN!
        }
    }

    return state_;
}


ChessStoneColor ChessEnv::GetWinner() const {
    int result = CalcResult();
    if (result == 3){
      return CHessComm::EMPTY;
    }else if (result == 1){
      return ChessComm::WHITE;
    }else if(result == 2){
      return ChessComm::BLACK;
    }
}

void ChessEnv::GetResult(float& res) {
	if (GetWinner() == ChessComm::EMPTY){
		res = 0.0f;
	}else{
        res = GetWinner() == CurrentPlayer() ? -1.0f : 1.0f;
    }
}

vector<bool> ChessEnv::GetFeature(BaseFeature& feature) const { // HWC
  feature.clear();
  feature.resize(CHESSBOARD_SIZE * (SIZE_HISTORYEACHSIDE + 1));
  //vector<bool> feature(CHESSBOARD_SIZE * (SIZE_HISTORYEACHSIDE + 1), 0);
  // because push white into history first,
  // if next player is white,
  // we should get planes swap of each two planes
  int reverse_plane = int(Self() == WHITE);
  for (ChessSize i = 0; i < SIZE_HISTORYEACHSIDE && i < feature_history_list_.size(); ++i) {
    const string &feature_str = *(feature_history_list_.rbegin() + (i ^ reverse_plane));
    for (int j = 0, k = i; j < CHESSBOARD_SIZE; ++j, k += (SIZE_HISTORYEACHSIDE + 1)) {
      feature[k] = feature_str[j] - '0';
    }
  }
  if (Self() == WHITE) {
    for (int j = 0, k = SIZE_HISTORYEACHSIDE; j < CHESSBOARD_SIZE; ++j, k += (SIZE_HISTORYEACHSIDE + 1)) {
      feature[k] = 1;
    }
  }
  return feature;
}

int ChessEnv::GetActionNum() {//include RESIGN
	return 1073741824 + 1;
}
 
void ChessEnv::GetInputDim(vector<int>& input_dim) {
    input_dim.clear();
    input_dim.push_back(SIZE_HISTORYEACHSIDE);
    input_dim.push_back(BORDER_SIZE);
    input_dim.push_back(BORDER_SIZE)
}

void ChessEnv::GetLegalAction(vector<int>& action) {
    action.clear();
    for (unsigned int i = 0; i < action_list_.size(); ++i){
        action.push_back(action_list_[i]);
    }
}

std::string ChessEnv::action2str(int action){
    ChessCoordId from_id, to_id;
    ChessStoneColor captured_stone, promoted_stone, piece;
    ChessFunction::ActionToId(action, piece, captured_stone, promoted_stone, from_id, to_id);
    char buffer[12];
    buffer[0] = 'f';
    buffer[1] = 'r';
    buffer[2] = 'o';
    buffer[3] = 'm';
    buffer[4] = ':';
    buffer[7] = 't';
    buffer[8] = 'o';
    buffer[9] = ':';
    ChessCoordId from_x, from_y, to_x, to_y;
    ChessFunction::IdToCoord(from_id, from_x, from_y);
    ChessFunction::IdToCoord(to_id, to_x, to_y);
    if (!InBoard(from_x, from_y)) {
        buffer[5] = buffer[6] = 'z';
    } else if(InBoard(from_x, from_y)){
        buffer[5] = from_x + 'a';
        buffer[6] = from_y + 'a';
    }
    if (!InBoard(to_x, to_y)) {
        buffer[10] = buffer[11] = 'z';
    } else if(InBoard(to_x, to_y)){
        buffer[10] = to_x + 'a';
        buffer[11] = to_y + 'a';
    }
    return string(buffer, 12);
}

/*void ChessEnv::TransformCoord(ChessCoordId &x, ChessCoordId &y, int mode, bool reverse = false)
{
    if (reverse) {
        if (mode & 4) std::swap(x, y);
        if (mode & 2) y = ChessComm::BORDER_SIZE - y - 1;
        if (mode & 1) x = ChessComm::BORDER_SIZE - x - 1;
    } else {
        if (mode & 1) x = ChessComm::BORDER_SIZE - x - 1;
        if (mode & 2) y = ChessComm::BORDER_SIZE - y - 1;
        if (mode & 4) std::swap(x, y);
    }
}*/
 
// void ChessEnv::TransformFeatures(BaseFeature& feature, int transform_mode) {
//     BaseFeature ret(feature.size());
//     int depth = features.size() / ChessComm::CHESSBOARD_SIZE;
//     for (int i = 0; i < ChessComm::CHESSBOARD_SIZE; ++i) {
//         ChessCoordId x, y;
//         ChessFunction::IdToCoord(i, x, y);
//         TransformCoord(x, y, mode);
//         int j = ChessFunction::CoordToId(x, y);
//         for (int k = 0; k < depth; ++k) {
//             ret[i * depth + k] = features[j * depth + k];
//         }
//     }
//     feature = std::move(ret);
// }

// void TransformPolicy(int transform_mode) {
// }


